package ru.yandex.chemodan.app.psbilling.core.billing.groups.payment;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.AutoResurrectionPayManager;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.GroupBillingService;
import ru.yandex.chemodan.app.psbilling.core.config.Settings;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupTrustPaymentRequestDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupTrustPaymentRequest;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.PaymentRequestStatus;
import ru.yandex.chemodan.balanceclient.exception.BalanceErrorCodeException;
import ru.yandex.chemodan.balanceclient.model.response.CheckRequestPaymentResponse;
import ru.yandex.inside.passport.PassportUid;

@Slf4j
@RequiredArgsConstructor
public class PaymentChecker {
    private final BalanceService balanceService;
    private final GroupDao groupDao;
    private final GroupTrustPaymentRequestDao groupTrustPaymentRequestDao;
    private final AutoResurrectionPayManager autoResurrectionPayManager;
    private final GroupBillingService groupBillingService;
    private final CardBindingChecker cardBindingChecker;
    private final Settings settings;

    public void checkPaymentRequestStatus(String requestId, String transactionId) {
        Option<GroupTrustPaymentRequest> paymentRequestO = groupTrustPaymentRequestDao.findByRequestId(requestId);
        if (!paymentRequestO.isPresent()) { // CHEMODAN-76494
            log.warn("The request {} is not found in DB. Check for problems in payment creation method '{}'",
                    requestId, "getTrustPaymentFormUrl");
            return;
        }

        checkPaymentRequestStatus(paymentRequestO.get(), Option.of(transactionId));
    }

    public void checkPaymentRequestStatus(String requestId) {
        Option<GroupTrustPaymentRequest> paymentRequestO = groupTrustPaymentRequestDao.findByRequestId(requestId);
        if (paymentRequestO.isEmpty()) { // CHEMODAN-76494
            log.warn("The request {} is not found in DB. Check for problems in payment creation method '{}'",
                    requestId, "getTrustPaymentFormUrl");
            return;
        }

        GroupTrustPaymentRequest paymentRequest = paymentRequestO.get();

        checkPaymentRequestStatus(paymentRequest, paymentRequest.getTransactionId());
    }

    public void checkPaymentRequestStatus(GroupTrustPaymentRequest paymentRequest, Option<String> transactionIdO) {
        String requestId = paymentRequest.getRequestId();
        if (!PaymentRequestStatus.INIT.equals(paymentRequest.getStatus())) {
            log.info("Status of request {} has been already updated ({})", requestId, paymentRequest.getStatus());
            return;
        }

        Option<CheckRequestPaymentResponse> paymentResponseO = getCheckRequestPaymentResponse(
                paymentRequest,
                transactionIdO,
                requestId
        );

        if (paymentResponseO.isEmpty()) {
            if (paymentRequest.getCreatedAt().plus(settings.getAcceptableGroupPaymentExpirationTime()).isBeforeNow()) {
                log.info("Payment request for uid={}; requestId={}; purchase_token={} is expired",
                        paymentRequest.getOperatorUid(), paymentRequest.getRequestId(),
                        paymentRequest.getTransactionId());
                updateStatus(paymentRequest, transactionIdO, requestId, PaymentRequestStatus.CANCELLED,
                        Option.of("Wait in init expired"));
            }

            return;
        }

        CheckRequestPaymentResponse paymentResponse = paymentResponseO.get();

        transactionIdO = transactionIdO
                .orElse(() -> Option.ofNullable(paymentResponse.getTransactionId()));

        PaymentRequestStatus requestStatus = getPaymentRequestStatus(paymentResponse);
        Option<String> error = Option.when(requestStatus == PaymentRequestStatus.CANCELLED,
                paymentResponse::getResponseCode);

        updateStatus(paymentRequest, transactionIdO, requestId, requestStatus, error);
    }

    private void updateStatus(
            GroupTrustPaymentRequest paymentRequest,
            Option<String> transactionIdO,
            String requestId,
            PaymentRequestStatus newPaymentRequestStatus,
            Option<String> error
    ) {
        log.info("payment status is {} for uid={}; requestId={}; purchase_token={}; " +
                        "GroupTrustPaymentRequest={}; error={}; transactionIdO={}", newPaymentRequestStatus,
                paymentRequest.getOperatorUid(), paymentRequest.getRequestId(), paymentRequest.getTransactionId(),
                paymentRequest, error, transactionIdO);

        if (PaymentRequestStatus.INIT.equals(newPaymentRequestStatus)) {
            log.info("The payment request status has not been update since creation requestId={}", requestId);
            return;
        }
        if (PaymentRequestStatus.SUCCESS.equals(newPaymentRequestStatus)) {
            String transactionId = transactionIdO
                    .orElseThrow(() -> new IllegalStateException(
                            String.format("Transaction_id not fount in group_pay_request=" + paymentRequest.getId())));

            handleSuccessPaymentRequestStatus(paymentRequest, transactionId);
        }

        transactionIdO.ifPresent(t -> groupTrustPaymentRequestDao.updateTransactionIdIfNull(paymentRequest.getId(), t));

        groupTrustPaymentRequestDao.updateStatusIfInInit(paymentRequest.withStatus(newPaymentRequestStatus)
                .withError(error));
    }

    private Option<CheckRequestPaymentResponse> getCheckRequestPaymentResponse(
            GroupTrustPaymentRequest paymentRequest,
            Option<String> transactionIdO,
            String requestId
    ) {
        try {
            CheckRequestPaymentResponse response = balanceService.checkRequestPayment(
                    requestId,
                    transactionIdO,
                    Long.parseLong(paymentRequest.getOperatorUid())
            );

            return Option.ofNullable(response);
        } catch (BalanceErrorCodeException e) {
            if (e.getCodes().containsTs(BalanceErrorCodeException.BalanceErrorCode.NO_PAYMENTS_FOR_REQUEST.name())) {
                log.info("Balance check without payment. All error codes {}", e.getCodes(), e);
                return Option.empty();
            }

            throw e;
        }
    }

    private PaymentRequestStatus getPaymentRequestStatus(CheckRequestPaymentResponse paymentResponse) {
        return Option.ofNullable(paymentResponse.getResponseCode())
                .map(code -> CheckRequestPaymentResponse.SUCCESS_PAYMENT_STATUS.equals(code)
                        ? PaymentRequestStatus.SUCCESS
                        : PaymentRequestStatus.CANCELLED)
                .orElse(PaymentRequestStatus.INIT);
    }

    private void handleSuccessPaymentRequestStatus(GroupTrustPaymentRequest paymentRequest, String transactionId) {

        ListF<Group> groups = groupDao.findGroupsByPaymentInfoClient(paymentRequest.getClientId());

        // костыль, пока не сделаем уникальный payment_info в отдельной таблице
        SetF<BalancePaymentInfo> paymentInfos = groups
                .map(x -> x.getPaymentInfo().get())
                .unique();

        if (paymentInfos.size() > 1) {
            log.warn("found several different payment infos by one client_id");
        }


        if (paymentRequest.getGroupServicesInfo().isPresent()) {
            if (paymentRequest.getMoney().isEmpty()) {
                log.error("money payment info not found for {}", paymentRequest);
            } else {
                autoResurrectionPayManager.resurrectServices(paymentRequest.getClientId(),
                        paymentRequest.getMoney().get().getCurrency(),
                        paymentRequest.getGroupServicesInfo().get());
            }
        }

        for (BalancePaymentInfo paymentInfo : paymentInfos) {
            groupBillingService.updateGroupsBillingStatus(
                    paymentInfo,
                    groups.filter(x -> x.getPaymentInfo().get().equals(paymentInfo)),
                    false);
        }

        Option<BalancePaymentInfo> paymentInfo = paymentInfos.iterator().nextO();
        PassportUid operatorUid = PassportUid.cons(Long.parseLong(paymentRequest.getOperatorUid()));
        cardBindingChecker.checkBindingAndSaveCard(operatorUid, transactionId, paymentInfo,
                paymentRequest.getCreatedAt());
    }
}
