package ru.yandex.chemodan.app.psbilling.core.billing.groups.tasks;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.LocalDate;

import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.ClientBalanceCalculator;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceTransactionsDao;
import ru.yandex.chemodan.app.psbilling.core.synchronization.PsBillingQueueNames;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.util.BatchFetchingUtils;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class UpdateClientTransactionsBalanceTask extends OnetimeTaskSupport<UpdateClientTransactionsBalanceTask.Parameters> {
    private static final Logger logger = LoggerFactory.getLogger(ClientBalanceCalculator.class);
    private static final int CLIENTS_BATCH_SIZE = 10000;
    private GroupServiceTransactionsDao groupServiceTransactionsDao;
    private TaskScheduler taskScheduler;

    public UpdateClientTransactionsBalanceTask(LocalDate transactionsDate) {
        super(new UpdateClientTransactionsBalanceTask.Parameters(transactionsDate));
    }

    public UpdateClientTransactionsBalanceTask(GroupServiceTransactionsDao groupServiceTransactionsDao,
                                               TaskScheduler taskScheduler) {
        super(UpdateClientTransactionsBalanceTask.Parameters.class);
        this.groupServiceTransactionsDao = groupServiceTransactionsDao;
        this.taskScheduler = taskScheduler;
    }

    @Override
    protected void execute(UpdateClientTransactionsBalanceTask.Parameters parameters, ExecutionContext context) {
        BatchFetchingUtils.<Long, Long>fetchAndProcessBatchedEntities(
                (batchSize, lastClientId) ->
                        groupServiceTransactionsDao.findTransactionsClients(
                                parameters.transactionsDate, lastClientId, CLIENTS_BATCH_SIZE),
                Function.identityF(),
                clientId -> taskScheduler.schedule(new UpdateClientBalanceTask(clientId)),
                CLIENTS_BATCH_SIZE,
                logger);
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return new RescheduleConstant(Duration.standardMinutes(1), 5);
    }

    @Override
    public TaskQueueName queueName() {
        return PsBillingQueueNames.CRON;
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(60);
    }

    @BenderBindAllFields
    @AllArgsConstructor
    @Data
    public static class Parameters {
        private final LocalDate transactionsDate;
    }
}
