package ru.yandex.chemodan.app.psbilling.core.billing.users;

import java.lang.reflect.Field;
import java.math.BigDecimal;

import com.google.common.base.Throwables;
import lombok.Builder;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.apache.commons.lang3.builder.StandardToStringStyle;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.joda.time.Instant;

import ru.yandex.chemodan.app.psbilling.core.billing.users.processors.CalculatedPrice;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Refund;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserServiceBillingStatus;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPrice;
import ru.yandex.chemodan.trust.client.responses.PaymentResponse;
import ru.yandex.chemodan.trust.client.responses.SubscriptionResponse;

@Slf4j
public class BillingActionsReportingService {
    private static final ToStringStyle REPORT_DATA_STYLE = new StandardToStringStyle() {{
        setUseClassName(false);
        setUseIdentityHashCode(false);
        setFieldSeparator(", ");
        setFieldNameValueSeparator(": ");
        setContentStart("");
        setContentEnd("");
    }};

    public enum Action {
        PROCESS_ORDER("process order"),
        BUY_NEW("buy_new"),
        UNSUBSCRIBE("unsubscribe"),
        PROLONG_AUTO("prolong_auto"),
        ORDER_UPGRADED("order_upgraded"),
        RESTORE_FROM_HOLD("restore_from_hold"),
        PROMOCODE_SPACE("promocode_space"),
        REFUND("refund");

        Action(String value) {
            this.value = value;
        }

        String value;

        @Override
        public String toString() {
            return value;
        }
    }


    @Builder
    @Getter
    @Setter
    public static class ReportData {
        Action action;
        @Builder.Default
        String status = "success";
        String statusCode;
        String statusDescription;
        String orderId; //trust order id
        String refundId;
        String uid;
        SubscriptionResponse subscription;
        String productCode;
        String period; //price.getPeriod().getPeriod() or subscription.getSubscriptionPeriod
        BigDecimal price;
        Boolean isStartPeriod;
        String currency;
        String oldOrderId;
        String oldServiceId;
        Instant oldExpirationDate;
        Instant newExpirationDate; // or subscribedUntil
        String errorCode;
        String errorMessage;
        String packageName;
        UserServiceBillingStatus serviceBillingStatus;
        String active_promo;

        private BillingActionsReportingService reporter;

        public static class ReportDataBuilder {
            public ReportDataBuilder order(Order order) {
                if (order != null) {
                    this.orderId(order.getTrustOrderId())
                            .uid(order.getUid())
                            .packageName(order.getPackageName().orElse((String) null));
                }
                return this;
            }

            public ReportDataBuilder userProduct(UserProduct product) {
                if (product != null) {
                    this.productCode(product.getCode());
                }
                return this;
            }

            public ReportDataBuilder userProductPrice(UserProductPrice price){
                if (price != null) {
                    this.period(price.getPeriod().getPeriod().toString())
                            .price(price.getPrice())
                            .currency(price.getCurrencyCode());
                }
                return this;
            }

            public ReportDataBuilder refund(Refund refund) {
                if (refund != null) {
                    this.refundId(refund.getTrustRefundId());
                }
                return this;
            }

            public ReportDataBuilder subscriptionResponse(SubscriptionResponse subscriptionResponse) {
                this.subscription(subscriptionResponse)
                        .period(subscriptionResponse.getSubscriptionPeriod())
                        .currency(subscriptionResponse.getCurrency().orElse((String) null));
                return this;
            }

            public ReportDataBuilder paymentResponse(PaymentResponse paymentResponse) {
                this.status(paymentResponse.getPaymentStatus().toString())
                        .statusCode(paymentResponse.getPaymentStatusCode())
                        .statusDescription(paymentResponse.getPaymentStatusDescription())
                        .price(paymentResponse.getCurrentAmount());
                return this;
            }

            public ReportDataBuilder calculatedPrice(CalculatedPrice calculatedPrice) {
                return this.price(calculatedPrice.getPrice())
                        .isStartPeriod(calculatedPrice.isStartPeriod());
            }

            public void finish() {
                reporter.logBillingAction(this.build());
            }
        }

        public String toString() {
            return new ReflectionToStringBuilder(this, REPORT_DATA_STYLE) {
                @Override
                protected boolean accept(Field field) {
                    if("reporter".equals(field.getName())){
                        return false;
                    }
                    try {
                        return field.get(getObject()) != null && super.accept(field);
                    } catch (IllegalAccessException e) {
                        throw Throwables.propagate(e);
                    }
                }

                @Override
                public ToStringBuilder append(String fieldName, Object obj) {
                    //camel case to snake case
                    fieldName = fieldName.replaceAll("([a-z])([A-Z]+)", "$1_$2").toLowerCase();
                    return super.append(fieldName, obj);
                }
            }.build();
        }
    }

    public void logBillingAction(ReportData reportData) {
        //inner method for asserts in tests
        log.info(composeReportString(reportData));
    }

    public ReportData.ReportDataBuilder builder(Action action) {
        return ReportData.builder().action(action).reporter(this);
    }

    protected String composeReportString(ReportData reportData) {
        return reportData.toString();
    }
}
