package ru.yandex.chemodan.app.psbilling.core.billing.users;

import java.util.UUID;

import javax.annotation.Nullable;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.billing.users.processors.OrderProcessorFacade;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderStatus;
import ru.yandex.chemodan.app.psbilling.core.synchronization.PsBillingQueueNames;
import ru.yandex.commune.bazinga.scheduler.ActiveUidBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUniqueIdentifierConverter;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;

public class CheckOrderTask extends OnetimeTaskSupport<CheckOrderTask.Parameters> {
    private OrderProcessorFacade orderProcessorFacade;

    public CheckOrderTask(OrderProcessorFacade orderProcessorFacade) {
        super(Parameters.class);
        this.orderProcessorFacade = orderProcessorFacade;
    }

    public CheckOrderTask(UUID orderId, OrderStatus orderStatus) {
        super(new Parameters(orderId.toString(), Option.of(orderStatus)));
    }

    @Override
    protected void execute(Parameters parameters, ExecutionContext context) throws Exception {
        UUID orderId = UUID.fromString(parameters.getOrderId());
        if (parameters.getOrderStatus().isPresent())
            orderProcessorFacade.processByOrderIdAndState(orderId, parameters.getOrderStatus().get());
        else
            orderProcessorFacade.processByOrderId(orderId); //legacy way for old tasks
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return new RescheduleConstant(Duration.standardMinutes(1), 5);
    }

    @Override
    public TaskQueueName queueName() {
        return PsBillingQueueNames.CHECK_TRUST_PAYMENTS;
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(5);
    }

    @Override
    public ActiveUidBehavior activeUidBehavior() {
        return new ActiveUidBehavior(ActiveUidDropType.WHEN_RUNNING, ActiveUidDuplicateBehavior.DO_NOTHING);
    }

    @Nullable
    @Override
    public Class<? extends ActiveUniqueIdentifierConverter<?, ?>> getActiveUidConverter() {
        return CheckOrderTask.Parameters.Converter.class;
    }


    @BenderBindAllFields
    @AllArgsConstructor
    @Data
    public static class Parameters {
        private final String orderId;
        private final Option<OrderStatus> orderStatus;

        public static class Converter implements
                ActiveUniqueIdentifierConverter<CheckOrderTask.Parameters, CheckOrderTask.Parameters>
        {
            @Override
            public Class<CheckOrderTask.Parameters> getActiveUniqueIdentifierClass() {
                return CheckOrderTask.Parameters.class;
            }

            @Override
            public CheckOrderTask.Parameters convert(CheckOrderTask.Parameters parameters) {
                return parameters;
            }
        }
    }
}
