package ru.yandex.chemodan.app.psbilling.core.billing.users;

import java.util.UUID;

import javax.annotation.Nullable;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.joda.time.Duration;

import ru.yandex.chemodan.app.psbilling.core.billing.users.processors.OrderProcessorFacade;
import ru.yandex.chemodan.app.psbilling.core.synchronization.PsBillingQueueNames;
import ru.yandex.chemodan.app.psbilling.core.users.UserService;
import ru.yandex.chemodan.app.psbilling.core.users.UserServiceManager;
import ru.yandex.commune.bazinga.scheduler.ActiveUidBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUniqueIdentifierConverter;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author friendlyevil
 */
public class CheckUserServiceTask extends OnetimeTaskSupport<CheckUserServiceTask.Parameters> {
    private final Logger logger = LoggerFactory.getLogger(CheckUserServiceTask.class);

    private UserServiceManager userServiceManager;
    private OrderProcessorFacade orderProcessorFacade;

    public CheckUserServiceTask(UserServiceManager userServiceManager, OrderProcessorFacade orderProcessorFacade) {
        super(Parameters.class);
        this.userServiceManager = userServiceManager;
        this.orderProcessorFacade = orderProcessorFacade;
    }

    public CheckUserServiceTask(UUID serviceId) {
        super(new Parameters(serviceId.toString()));
    }

    @Override
    protected void execute(Parameters parameters, ExecutionContext context) throws Exception {
        UserService userService = userServiceManager.findById(UUID.fromString(parameters.getServiceId()));

        if (userService.getLastPaymentOrderId().isPresent()) {
            orderProcessorFacade.processByOrderId(userService.getLastPaymentOrderId().get());
        } else if (userService.getNextCheckDate().isPresent() && userService.getNextCheckDate().get().isBeforeNow()) {
            logger.info("Service without order will be disabled: {}", userService);
            userServiceManager.disableService(userService.getId());
        }
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(1);
    }

    @Override
    public TaskQueueName queueName() {
        return PsBillingQueueNames.CHECK_TRUST_PAYMENTS;
    }

    @Override
    public ActiveUidBehavior activeUidBehavior() {
        return new ActiveUidBehavior(ActiveUidDropType.WHEN_RUNNING, ActiveUidDuplicateBehavior.DO_NOTHING);
    }

    @Nullable
    @Override
    public Class<? extends ActiveUniqueIdentifierConverter<?, ?>> getActiveUidConverter() {
        return CheckUserServiceTask.Parameters.Converter.class;
    }

    @BenderBindAllFields
    @AllArgsConstructor
    @Data
    public static class Parameters {
        private final String serviceId;

        public static class Converter implements ActiveUniqueIdentifierConverter<CheckUserServiceTask.Parameters,
                CheckUserServiceTask.Parameters> {
            @Override
            public Class<CheckUserServiceTask.Parameters> getActiveUniqueIdentifierClass() {
                return CheckUserServiceTask.Parameters.class;
            }

            @Override
            public CheckUserServiceTask.Parameters convert(CheckUserServiceTask.Parameters parameters) {
                return parameters;
            }
        }
    }
}
