package ru.yandex.chemodan.app.psbilling.core.billing.users;

import java.math.BigDecimal;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.function.Function;

import lombok.AllArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriod;
import ru.yandex.chemodan.app.psbilling.core.entities.products.BillingType;
import ru.yandex.chemodan.app.psbilling.core.products.TrialDefinition;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPeriod;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPrice;
import ru.yandex.chemodan.app.psbilling.core.texts.TankerTranslation;
import ru.yandex.chemodan.trust.client.TrustClient;
import ru.yandex.chemodan.trust.client.requests.CreateProductRequest;
import ru.yandex.chemodan.trust.client.requests.LocalName;
import ru.yandex.chemodan.trust.client.requests.Price;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class ProductsExportService {
    private static final Logger logger = LoggerFactory.getLogger(ProductsExportService.class);
    public final static SetF<BillingType> BILLING_TYPES_TO_EXPORT = Cf.set(BillingType.values())
            .filter(BillingType::isInappProduct)
            .plus(BillingType.TRUST);

    private final TrustClient trustClient;
    private final UserProductManager userProductManager;
    private final DynamicProperty<Boolean> enableInappExport =
            new DynamicProperty<>("ps-billing.enable-inapp-products-export", true);

    public void exportProducts() {
        userProductManager.findByBillingTypes(BILLING_TYPES_TO_EXPORT.toArray(BillingType.class))
                .forEach(this::exportProduct);
    }


    void exportProduct(UserProduct userProduct) {
        Validate.isTrue(BILLING_TYPES_TO_EXPORT.containsTs(userProduct.getBillingType()));
        logger.info("Exporting product {} to trust", userProduct);

        if (userProduct.getProductPrices().isEmpty()) {
            logger.warn("Unable to export product {} without prices", userProduct);
            return;
        }

        if (userProduct.getBillingType() == BillingType.TRUST) {
            exportSimpleProduct(userProduct);
            if (userProduct.isAllowAutoProlong()) {
                exportSubscriptionProduct(userProduct);
            }
            return;
        }

        if (userProduct.getBillingType().isInappProduct() && enableInappExport.get()) {
            exportInappProduct(userProduct);
        }
    }

    private void exportInappProduct(UserProduct userProduct) {
        for (Tuple2<UserProductPeriod, ListF<UserProductPrice>> entry : userProduct.getProductPrices().entries()) {
            UserProductPeriod period = entry._1;
            String productId = period.getTrustProductId(true);
            CreateProductRequest.CreateProductRequestBuilder builder = CreateProductRequest.builder()
                    .name(productId)
                    .productId(productId)
                    .productType(userProduct.getBillingType().getTrustProductType(true))
                    .localNames(toLocalNames(userProduct.getTitle()))
                    .prices(mapPrices(period.getPrices()))
                    .trustServiceId(userProduct.getTrustServiceId()
                            .orElseThrow(() -> new NoSuchElementException("trust service id not defined for " + userProduct)));

            period.getPackageName().ifPresent(builder::packageName);

            //см. https://a.yandex-team.ru/arc_vcs/disk/mpfs/lib/mpfs/core/billing/processing/marketing
            // .py?rev=6adf322125fa7cca79d3dc937dbd742ac83688d6#L88
            //для инаппов это все не поддерживается
            //            Validate.none(userProduct.getTrialDefinition());
            //            Validate.none(userProduct.getTrustSubsGracePeriod());
            //            Validate.none(userProduct.getTrustSubsChargingRetryDelay());
            //            Validate.none(userProduct.getTrustSubsChargingRetryLimit());

            trustClient.createProduct(builder.build());
        }
    }

    private void exportSubscriptionProduct(UserProduct userProduct) {
        for (UserProductPeriod period : userProduct.getProductPrices().keys()) {
            CreateProductRequest.CreateProductRequestBuilder builder = initBuilder(period, true)
                    .parentProductId(period.getTrustProductId(false))
                    .subscriptionPeriod(period.getPeriod().toTrustPeriod());

            userProduct.getTrustSubsGracePeriod().ifPresent(builder::subscriptionGracePeriod);
            userProduct.getTrustSubsChargingRetryDelay().ifPresent(builder::subscriptionRetryChargingDelay);
            userProduct.getTrustSubsChargingRetryLimit().ifPresent(builder::subscriptionRetryChargingLimit);
            if (period.hasStartPeriod()) {
                ListF<UserProductPrice> prices = period.getPrices();
                prices.forEach(price -> Validate.isTrue(price.getStartPeriodPrice().isPresent(),
                        "Period " + period.getCode() + " have start period but there are price without start period " +
                                "price"));
                String subscriptionStartPeriod = period.getStartPeriodDuration()
                        .map(CustomPeriod::toTrustPeriod)
                        .get();

                builder.subscriptionStartPeriodCount(period.getStartPeriodCount().get())
                        .subscriptionStartPeriod(subscriptionStartPeriod)
                        .startPeriodPrices(mapPrices(prices, price -> price.getStartPeriodPrice().get()))
                        .singlePurchase(1);
            }

            //поддерживаются только триалы с 0 стоимостью и с фиксированным периодом
            userProduct.getTrialDefinition().ifPresent(TrialDefinition::ensurePriceZero);
            userProduct.getTrialDefinition().ifPresent(def -> builder
                    .subscriptionTrialPeriod(def.getDurationOrThrow().toTrustPeriod())
                    .singlePurchase(1).subsTrialRepeat(1));

            trustClient.createProduct(builder.build());
        }
    }

    private void exportSimpleProduct(UserProduct userProduct) {
        for (UserProductPeriod period : userProduct.getProductPrices().keys()) {
            CreateProductRequest.CreateProductRequestBuilder builder = initBuilder(period, false);
            trustClient.createProduct(builder.build());
        }
    }

    private static CreateProductRequest.CreateProductRequestBuilder initBuilder(
            UserProductPeriod period, boolean autoprolong) {
        String productId = period.getTrustProductId(autoprolong);
        UserProduct userProduct = period.getUserProduct();
        return CreateProductRequest.builder()
                .name(productId)
                .productId(productId)
                .productType(userProduct.getBillingType().getTrustProductType(autoprolong))
                .localNames(toLocalNames(userProduct.getTitle()))
                .prices(mapPrices(period.getPrices()))
                .fiscalNds("nds_20_120")
                .trustServiceId(
                        userProduct.getTrustServiceId().orElseThrow(
                                () -> new NoSuchElementException("trust service id not defined for " + userProduct)))
                .fiscalTitle(period.getTrustFiscalTitle().orElseThrow(
                        () -> new NoSuchElementException("For trust product period not defined fiscal title" + period)));
    }

    private static List<Price> mapPrices(ListF<UserProductPrice> prices) {
        return mapPrices(prices, UserProductPrice::getPrice);
    }

    private static List<Price> mapPrices(ListF<UserProductPrice> prices,
                                         Function<UserProductPrice, BigDecimal> priceMapper) {
        Instant now = Instant.now();
        return prices.map(p -> new Price(p.getRegionId(), now, priceMapper.apply(p), p.getCurrencyCode()));
    }

    private static List<LocalName> toLocalNames(Option<TankerTranslation> translation) {
        return translation.map(TankerTranslation::getAllLanguageTexts).orElse(Cf.map())
                .mapEntries(LocalName::new);
    }

}
