package ru.yandex.chemodan.app.psbilling.core.billing.users;

import java.util.Objects;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.billing.users.processors.OrderProcessorFacade;
import ru.yandex.chemodan.app.psbilling.core.billing.users.processors.UnknownProductException;
import ru.yandex.chemodan.app.psbilling.core.entities.InappStore;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.ReceiptProcessResult;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsClientImpl;
import ru.yandex.chemodan.mpfs.UserBlockedException;
import ru.yandex.chemodan.mpfs.UserNotInitializedException;
import ru.yandex.chemodan.trust.client.responses.InappSubscription;
import ru.yandex.chemodan.trust.client.responses.InappSubscriptionState;
import ru.yandex.chemodan.trust.client.responses.ProcessInappReceiptResponse;
import ru.yandex.chemodan.trust.client.responses.ReceiptSubscriptionItem;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static ru.yandex.chemodan.trust.client.responses.ReceiptSubscriptionItem.SYNC_STATUS_SUCCESS;

@AllArgsConstructor
public class ReceiptHandlerV1 implements ReceiptHandler {
    private static final Logger logger = LoggerFactory.getLogger(ReceiptHandlerV1.class);
    private final OrderProcessorFacade orderProcessorFacade;
    private final MpfsClient mpfsClient;

    public ReceiptProcessResult processInappReceipt(PassportUid uid, InappStore inappStore, String currency,
                                                    Option<String> receiptO, String packageName) {
        ProcessInappReceiptResponse response = orderProcessorFacade.checkInappReceipt(
                uid, inappStore, currency, receiptO.getOrThrow("empty receipt"));

        ListF<Order> result = Cf.arrayList();
        //одновременно может приходить и отключение и включение, поэтому сортируем так, чтобы сначала отключались
        A3ExceptionWithStatus lastError = null;
        boolean needSendToMpfs = false;
        boolean mpfsServiceNotFinished = false;
        for (ReceiptSubscriptionItem item : sortInappSubscriptions(response.getItems())) {
            try {
                if (!Objects.equals(item.getSyncStatus(), SYNC_STATUS_SUCCESS)) {
                    logger.error("Problem in inapp subscription syncing {}", item);
                    throw new A3ExceptionWithStatus("syncing-error", HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
                }
                InappSubscription subscription = item.getSubscription();
                Option<Order> orderO = orderProcessorFacade.findOrder(subscription.getSubscriptionId());
                Order order = orderProcessorFacade.processInappSubscription(subscription, uid, packageName, orderO);

                // поверяем только для активных подписок, сравниваем uid пришедший и синхронизиованного оредера, потому
                // что для аппстора мы хитро храним uid заказа, из-за того что в рамках одного сторовского аккаунта для
                // всех подписок один и та же айдишка ордера, поэтому мы у себя сохраняем а ордер uid того, кто
                // создал услугу
                if (!Objects.equals(uid.toString(), order.getUid()) && subscription.getState() != InappSubscriptionState.FINISHED) {
                    logger.error("Uid in request: {}, uid in subscription: {}, order uid: {}",
                            uid, subscription.getUid(), order.getUid());
                    throw new A3ExceptionWithStatus("uid-mismatch", HttpStatus.SC_400_BAD_REQUEST);
                }

                if (subscription.getState() != InappSubscriptionState.FINISHED) {
                    result.add(order);
                }
            } catch (UnknownProductException e) {
                logger.warn("Found unknown product {} in receipt, receipt will be sent to mpfs", e.getProduct());
                needSendToMpfs = true;
                mpfsServiceNotFinished = item.getSubscription().getState() != InappSubscriptionState.FINISHED;
            } catch (Exception e) {
                logger.error("Failed to process item {}", item, e);
                lastError = e instanceof A3ExceptionWithStatus ? (A3ExceptionWithStatus) e :
                        new A3ExceptionWithStatus("receipt-failed", e, HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
            }
        }

        if (needSendToMpfs) {
            sendReceiptToMpfs(uid, mpfsServiceNotFinished, inappStore, currency, receiptO.get(), packageName);
        }

        if (lastError != null) {
            throw lastError;
        }

        return new ReceiptProcessResult(result.reverse()); // чтобы сначала были те, что активные
    }

    private void sendReceiptToMpfs(PassportUid uid, boolean mpfsServiceNotFinished, InappStore inappStoreType,
                                   String currency, String receipt, String packageName) {
        logger.info("sending receipt to mpfs");
        try {
            MpfsClientImpl.suppressByCodes(Cf.list(400), () -> mpfsClient.processInappReceipt(uid, packageName,
                    inappStoreType.toTrustType().getMpfsValue(), currency, receipt));
        } catch (UserNotInitializedException | UserBlockedException e) {
            throw new A3ExceptionWithStatus("user_not_active", HttpStatus.SC_400_BAD_REQUEST);
        }
    }

    private ListF<ReceiptSubscriptionItem> sortInappSubscriptions(ListF<ReceiptSubscriptionItem> items) {
        //одновременно может приходить и отключение и включение, поэтому сортируем так, чтобы сначала отключались
        ListF<ReceiptSubscriptionItem> result = Cf.toArrayList(
                items.filter(item -> item.getSubscription() != null && item.getSubscription().getState() == InappSubscriptionState.FINISHED));
        result.addAll(
                items.filter(item -> item.getSubscription() == null || item.getSubscription().getState() != InappSubscriptionState.FINISHED));
        return result;
    }
}
