package ru.yandex.chemodan.app.psbilling.core.billing.users;

import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import lombok.SneakyThrows;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function4;
import ru.yandex.chemodan.app.psbilling.core.billing.users.processors.OrderProcessorFacade;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.entities.InappStore;
import ru.yandex.chemodan.app.psbilling.core.entities.products.BillingType;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.ReceiptProcessResult;
import ru.yandex.chemodan.app.psbilling.core.groups.TrialService;
import ru.yandex.chemodan.app.psbilling.core.products.TrialDefinition;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPeriod;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.users.UserService;
import ru.yandex.chemodan.app.psbilling.core.users.UserServiceManager;
import ru.yandex.chemodan.concurrent.ExecutorUtils;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsClientImpl;
import ru.yandex.chemodan.mpfs.UserBlockedException;
import ru.yandex.chemodan.mpfs.UserNotInitializedException;
import ru.yandex.chemodan.trust.client.responses.AppleReceiptResponse;
import ru.yandex.chemodan.trust.client.responses.InappSubscription;
import ru.yandex.chemodan.trust.client.responses.InappSubscriptionState;
import ru.yandex.chemodan.trust.client.responses.ProcessInappReceiptResponse;
import ru.yandex.chemodan.trust.client.responses.ReceiptSubscriptionItem;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static ru.yandex.chemodan.trust.client.responses.ReceiptSubscriptionItem.SYNC_STATUS_SUCCESS;

public class ReceiptHandlerV2 implements ReceiptHandler {
    private static final Logger logger = LoggerFactory.getLogger(ReceiptHandlerV2.class);

    private final UserServiceManager userServiceManager;
    private final UserProductManager userProductManager;
    private final OrderProcessorFacade orderProcessorFacade;
    private final MpfsClient mpfsClient;
    @SuppressWarnings("unused")
    private final FeatureFlags featureFlags;
    private final ThreadPoolExecutor receiptCheckerExecutor;
    private final TrialService trialService;


    public ReceiptHandlerV2(UserServiceManager userServiceManager, UserProductManager userProductManager,
                            OrderProcessorFacade orderProcessorFacade, MpfsClient mpfsClient,
                            FeatureFlags featureFlags, int receiptCheckerMaxPoolSize, TrialService trialService) {
        this.userServiceManager = userServiceManager;
        this.userProductManager = userProductManager;
        this.orderProcessorFacade = orderProcessorFacade;
        this.mpfsClient = mpfsClient;
        this.featureFlags = featureFlags;
        this.receiptCheckerExecutor = new ThreadPoolExecutor(10, receiptCheckerMaxPoolSize, 60,
                TimeUnit.SECONDS, new SynchronousQueue<>(), new ThreadPoolExecutor.CallerRunsPolicy());
        this.trialService = trialService;
    }

    // https://wiki.yandex-team.ru/users/mrdaemon/projects/mail360purchases/multivodstvo/
    private final MapF<Tuple2<ReceiptState, BoughtProductState>, Function4<PassportUid, String,
            Option<InappSubscription>, Option<Order>, Option<Order>>> receiptHandleDecisions =
            Cf.toHashMap(Cf.list(
                    Tuple2.tuple(Tuple2.tuple(ReceiptState.NONE, BoughtProductState.NONE),
                            this::doNothing),
                    Tuple2.tuple(Tuple2.tuple(ReceiptState.NONE, BoughtProductState.SAME_APPLICATION),
                            this::errorOtherStoreAccount),
                    Tuple2.tuple(Tuple2.tuple(ReceiptState.NONE, BoughtProductState.OTHER_APPLICATION),
                            this::errorOtherApplication),

                    Tuple2.tuple(Tuple2.tuple(ReceiptState.NEW, BoughtProductState.NONE),
                            this::boundReceipt),
                    Tuple2.tuple(Tuple2.tuple(ReceiptState.NEW, BoughtProductState.SAME_APPLICATION),
                            this::errorOtherStoreAccount),
                    Tuple2.tuple(Tuple2.tuple(ReceiptState.NEW, BoughtProductState.OTHER_APPLICATION),
                            this::errorOtherApplication),

                    Tuple2.tuple(Tuple2.tuple(ReceiptState.BOUND_TO_CURRENT_USER, BoughtProductState.NONE),
                            this::errorImpossibleState),
                    Tuple2.tuple(Tuple2.tuple(ReceiptState.BOUND_TO_CURRENT_USER, BoughtProductState.SAME_APPLICATION),
                            this::boundReceipt),
                    Tuple2.tuple(Tuple2.tuple(ReceiptState.BOUND_TO_CURRENT_USER, BoughtProductState.OTHER_APPLICATION),
                            this::errorImpossibleState),

                    Tuple2.tuple(Tuple2.tuple(ReceiptState.BOUND_TO_OTHER_USER, BoughtProductState.NONE),
                            this::errorOtherYandexAccount),
                    Tuple2.tuple(Tuple2.tuple(ReceiptState.BOUND_TO_OTHER_USER, BoughtProductState.SAME_APPLICATION),
                            this::errorOtherStoreAccount),
                    Tuple2.tuple(Tuple2.tuple(ReceiptState.BOUND_TO_OTHER_USER, BoughtProductState.OTHER_APPLICATION),
                            this::errorOtherApplication)
            ));

    public ReceiptProcessResult processInappReceipt(PassportUid uid, InappStore inappStore, String currency,
                                                    Option<String> receiptO, String packageName) {
        logger.info("processInappReceipt. uid: {}; inappStore: {}; currency: {}; receipt length: {}; packageName: {} ",
                uid, inappStore, currency, receiptO.map(String::length).orElse(0), packageName);
        if (!receiptO.isPresent()) {
            Option<Order> order = processSubscription(uid, inappStore, packageName, Option.empty());
            if (order.isPresent()) {
                return new ReceiptProcessResult(Cf.list(order.get()), Option.of(false));
            }
            return new ReceiptProcessResult(Cf.list(), Option.of(false));
        }
        ListF<Order> orders = Cf.arrayList();
        String receipt = receiptO.get();
        Future<Option<Boolean>> trialUsed = isTrialUsed(uid, inappStore, receipt);

        try {
            ProcessInappReceiptResponse response = orderProcessorFacade.checkInappReceipt(uid, inappStore,
                    currency, receipt);

            //одновременно может приходить и отключение и включение, поэтому сортируем так, чтобы сначала отключались
            A3ExceptionWithStatus lastError = null;
            boolean needSendToMpfs = false;
            for (ReceiptSubscriptionItem item : sortInappSubscriptions(response.getItems())) {
                try {
                    if (!Objects.equals(item.getSyncStatus(), SYNC_STATUS_SUCCESS)) {
                        logger.error("Problem in inapp subscription syncing {}", item);
                        throw new A3ExceptionWithStatus("syncing-error", HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
                    }
                    InappSubscription subscription = item.getSubscription();
                    Option<UserProductPeriod> period =
                            userProductManager.findPeriodByCodeO(subscription.getProductId());
                    if (!period.isPresent()) {
                        needSendToMpfs = true;
                        continue;
                    }
                    Option<Order> order = processSubscription(uid, inappStore, packageName,
                            Option.of(subscription));
                    if (order.isPresent()) {
                        orders.add(order.get());
                    }
                } catch (Exception e) {
                    logger.error("Failed to process item {}", item, e);
                    lastError = e instanceof A3ExceptionWithStatus ? (A3ExceptionWithStatus) e :
                            new A3ExceptionWithStatus("receipt-failed", e, HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
                }
            }

            if (needSendToMpfs) {
                sendReceiptToMpfs(uid, inappStore, currency, receipt, packageName);
            }

            if (lastError != null) {
                throw lastError;
            }
            try {
                return new ReceiptProcessResult(orders.reverse(), trialUsed.get()); //чтобы сначала были те, что
                // активные
            } catch (InterruptedException e) {
                // Interrupted exception may mean only application shutdown,
                // so doesn't matter what exception will be thrown here in this case
                throw new RuntimeException(e);
            } catch (ExecutionException e) {
                throw (RuntimeException) e.getCause(); //isTrialUsed can throw only runtimeException
            }
        } catch (Exception e) {
            trialUsed.cancel(true);
            throw e;
        }
    }

    private Future<Option<Boolean>> isTrialUsed(PassportUid uid, InappStore inappStore, String receipt) {
        switch (inappStore) {
            case APPLE_APPSTORE:
                return ExecutorUtils.submitWithYcridForwarding(() -> isAppstoreTrialUsed(receipt),
                        receiptCheckerExecutor);
            case GOOGLE_PLAY:
                return CompletableFuture.completedFuture(Option.of(isGooglePlayTrialUsed(uid))); //Trial usage status
            // is not used for google play. Add if required
            default:
                logger.error("Unexpected inappStore {}" + inappStore);
                return CompletableFuture.completedFuture(Option.empty());
        }
    }

    private Option<Boolean> isAppstoreTrialUsed(String receipt) {
        logger.info("Receipt checker pool size " + receiptCheckerExecutor.getPoolSize() + " of " + receiptCheckerExecutor.getMaximumPoolSize());
        AppleReceiptResponse rawReceipt = orderProcessorFacade.checkAppstoreReceipt(receipt);
        ListF<AppleReceiptResponse.InAppPurchase> purchases = Optional.ofNullable(rawReceipt)
                .map(AppleReceiptResponse::getResult)
                .map(AppleReceiptResponse.Result::getReceiptInfo)
                .map(AppleReceiptResponse.ReceiptInfo::getReceipt)
                .map(AppleReceiptResponse.Receipt::getInAppPurchases).orElse(Cf.list());
        if (purchases.isEmpty()) {
            logger.error("Trust answer have no info about inapp purchases. Fallback to trialUsed by default. " +
                    "Trust answer: {} ", rawReceipt);
            return Option.empty();
        }
        return Option.of(purchases.stream()
                .filter(inApp -> inApp.getIsInIntroOfferPeriod() || inApp.getIsTrialPeriod())
                .anyMatch(inApp -> userProductManager.findPeriodByCodeO(inApp.getProductId()).isPresent()));

    }

    private boolean isGooglePlayTrialUsed(PassportUid uid) {
        Set<String> trialComparisonKeys = userProductManager.findByBillingTypes(BillingType.INAPP_GOOGLE).stream()
                .flatMap(p -> p.getTrialDefinition()
                        .flatMapO(TrialDefinition::getSingleUsageComparisonKey)
                        .stream())
                .collect(Collectors.toSet());
        for (String trialComparisonKey : trialComparisonKeys) {
            if (!trialService.findUsageForUser(trialComparisonKey, uid).isEmpty()) {
                return true;
            }
        }
        return false;
    }

    private Option<Order> processSubscription(PassportUid uid,
                                              InappStore inappStore,
                                              String packageName,
                                              Option<InappSubscription> subscriptionO) {
        ReceiptState receiptState;
        Option<Order> order = Option.empty();
        if (subscriptionO.isPresent()) {
            InappSubscription subscription = subscriptionO.get();
            order = orderProcessorFacade.findOrder(subscription.getSubscriptionId());

            if (subscription.getState() == InappSubscriptionState.FINISHED
                    || subscription.getState() == InappSubscriptionState.ON_HOLD) {
                // не нужно дополнительной логики - тут будет отключение услуги
                orderProcessorFacade.processInappSubscription(subscription, uid, packageName, order);
                return Option.empty();
            }

            Option<UserService> userService = order.isPresent() && order.get().getUserServiceId().isPresent()
                    ? Option.of(userServiceManager.findById(order.get().getUserServiceId().get()))
                    : Option.empty();
            logger.info("order: {}; userService: {};", order, userService);
            if (!order.isPresent() || !userService.isPresent() || userService.get().getTarget() == Target.DISABLED) {
                receiptState = ReceiptState.NEW;
            } else if (order.get().getUid().equals(uid.toString())) {
                receiptState = ReceiptState.BOUND_TO_CURRENT_USER;
            } else {
                receiptState = ReceiptState.BOUND_TO_OTHER_USER;
            }
        } else {
            receiptState = ReceiptState.NONE;
        }
        BoughtProductState boughtProductState = getBoughtProductState(uid, inappStore, packageName, subscriptionO);
        logger.info("receiptState={} boughtProductState={}", receiptState, boughtProductState);
        return receiptHandleDecisions.getTs(Tuple2.tuple(receiptState, boughtProductState))
                .apply(uid, packageName, subscriptionO, order);
    }

    private BoughtProductState getBoughtProductState(PassportUid uid,
                                                     InappStore inappStore, String packageName,
                                                     Option<InappSubscription> subscriptionO) {
        ListF<UserService> existServices;
        if (subscriptionO.isPresent() && featureFlags.getAllowUserGetBoughtInappProduct().isEnabledForUid(uid)) {
            UserProduct subscriptionProduct =
                    userProductManager.findPeriodByCode(subscriptionO.get().getProductId()).getUserProduct();
            existServices = userServiceManager.findEnabled(uid.toString(), Option.empty())
                    .filter(userService -> userService.getUserProduct().conflictsWith(subscriptionProduct.getId()));
        } else {
            existServices = userServiceManager.findEnabled(uid.toString(), Option.empty())
                    .filter(x -> !x.getUserProduct().getBillingType().equals(BillingType.GROUP));
        }

        if (existServices.isEmpty()) {
            return BoughtProductState.NONE;
        }

        if (existServices.stream().anyMatch(x -> x.getUserProduct().getBillingType().equals(inappStore.getBillingType())
                && packageName.equals(x.getPackageName().orElse("")))) {
            return BoughtProductState.SAME_APPLICATION;
        }

        return BoughtProductState.OTHER_APPLICATION;
    }

    private ListF<ReceiptSubscriptionItem> sortInappSubscriptions(ListF<ReceiptSubscriptionItem> items) {
        //одновременно может приходить и отключение и включение, поэтому сортируем так, чтобы сначала отключались
        ListF<ReceiptSubscriptionItem> result = Cf.toArrayList(
                items.filter(item -> item.getSubscription() != null && item.getSubscription().getState() == InappSubscriptionState.FINISHED));
        result.addAll(
                items.filter(item -> item.getSubscription() == null || item.getSubscription().getState() != InappSubscriptionState.FINISHED));
        return result;
    }

    private void sendReceiptToMpfs(PassportUid uid, InappStore inappStore,
                                   String currency, String receipt, String packageName) {
        logger.info("sending receipt to mpfs");
        try {
            MpfsClientImpl.suppressByCodes(Cf.list(400), () -> mpfsClient.processInappReceipt(uid, packageName,
                    inappStore.toTrustType().getMpfsValue(), currency, receipt));
        } catch (UserNotInitializedException | UserBlockedException e) {
            throw new A3ExceptionWithStatus("user_not_active", HttpStatus.SC_400_BAD_REQUEST);
        }
    }

    @SneakyThrows
    private Option<Order> boundReceipt(PassportUid uid, String packageName, Option<InappSubscription> subscription,
                                       Option<Order> analyzedOrder) {
        logger.info("boundReceipt: uid={} packageName={} subscription={}", uid, packageName, subscription);
        return Option.of(orderProcessorFacade.processInappSubscription(subscription.get(), uid, packageName,
                analyzedOrder));
    }

    private Option<Order> doNothing(PassportUid uid, String packageName, Option<InappSubscription> subscription,
                                    Option<Order> analyzedOrder) {
        logger.info("doNothing: uid={} packageName={} subscription={}", uid, packageName, subscription);
        if (subscription.isPresent()) {
            return orderProcessorFacade.findOrder(subscription.get().getSubscriptionId());
        }
        return Option.empty();
    }

    private Option<Order> errorOtherStoreAccount(PassportUid uid, String packageName,
                                                 Option<InappSubscription> subscription, Option<Order> analyzedOrder) {
        logger.info("errorOtherStoreAccount: uid={} packageName={} subscription={}", uid, packageName, subscription);
        throw new A3ExceptionWithStatus("multiuser_store", HttpStatus.SC_400_BAD_REQUEST);
    }

    private Option<Order> errorOtherApplication(PassportUid uid, String packageName,
                                                Option<InappSubscription> subscription, Option<Order> analyzedOrder) {
        logger.info("errorOtherApplication: uid={} packageName={} subscription={}", uid, packageName, subscription);
        throw new A3ExceptionWithStatus("wrong_app", HttpStatus.SC_400_BAD_REQUEST);
    }

    private Option<Order> errorImpossibleState(PassportUid uid, String packageName,
                                               Option<InappSubscription> subscription, Option<Order> analyzedOrder) {
        logger.info("errorImpossibleState: uid={} packageName={} subscription={}", uid, packageName, subscription);
        throw new A3ExceptionWithStatus("impossible_state", HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
    }

    private Option<Order> errorOtherYandexAccount(PassportUid uid, String packageName,
                                                  Option<InappSubscription> subscription, Option<Order> analyzedOrder) {
        logger.info("errorOtherYandexAccount: uid={} packageName={} subscription={}", uid, packageName, subscription);
        throw new A3ExceptionWithStatus("multiuser_yandex", HttpStatus.SC_400_BAD_REQUEST);
    }

    private enum ReceiptState {
        NONE,
        NEW,
        BOUND_TO_CURRENT_USER,
        BOUND_TO_OTHER_USER,
    }

    private enum BoughtProductState {
        NONE,
        SAME_APPLICATION,
        OTHER_APPLICATION
    }
}
