package ru.yandex.chemodan.app.psbilling.core.billing.users;

import java.util.UUID;

import javax.annotation.Nullable;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.users.OrderDao;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Refund;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.synchronization.PsBillingQueueNames;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.commune.bazinga.scheduler.ActiveUidBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUniqueIdentifierConverter;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;

public class RefundOrderTask extends OnetimeTaskSupport<RefundOrderTask.Parameters> {
    private final UserBillingService userBillingService;
    private final TaskScheduler taskScheduler;
    private final OrderDao orderDao;


    public RefundOrderTask(UserBillingService userBillingService, TaskScheduler taskScheduler, OrderDao orderDao) {
        super(RefundOrderTask.Parameters.class);
        this.userBillingService = userBillingService;
        this.taskScheduler = taskScheduler;
        this.orderDao = orderDao;
    }

    public RefundOrderTask(UUID orderId) {
        super(new RefundOrderTask.Parameters(orderId.toString()));
        userBillingService = null;
        taskScheduler = null;
        orderDao = null;
    }

    @Override
    protected void execute(RefundOrderTask.Parameters parameters, ExecutionContext context) {
        Order order = orderDao.findById(UUID.fromString(parameters.getOrderId()));
        Option<Refund> refund_payment = userBillingService.initRefund(order, "Refund payment");
        if (refund_payment.isPresent()) {
            taskScheduler.scheduleLocalizedEmailTask(
                    "subscription_refund_started",
                    MailContext.builder()
                            .to(PassportUid.cons(Long.parseLong(order.getUid())))
                            .build()
            );
        }
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return new RescheduleConstant(Duration.standardMinutes(1), Integer.MAX_VALUE);
    }

    @Override
    public TaskQueueName queueName() {
        return PsBillingQueueNames.CHECK_TRUST_PAYMENTS;
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(5);
    }

    @Override
    public ActiveUidBehavior activeUidBehavior() {
        return new ActiveUidBehavior(ActiveUidDropType.WHEN_RUNNING, ActiveUidDuplicateBehavior.DO_NOTHING);
    }

    @Nullable
    @Override
    public Class<? extends ActiveUniqueIdentifierConverter<?, ?>> getActiveUidConverter() {
        return RefundOrderTask.Parameters.Converter.class;
    }


    @BenderBindAllFields
    @AllArgsConstructor
    @Data
    public static class Parameters {
        private final String orderId;

        public static class Converter implements
                ActiveUniqueIdentifierConverter<RefundOrderTask.Parameters, RefundOrderTask.Parameters> {
            @Override
            public Class<RefundOrderTask.Parameters> getActiveUniqueIdentifierClass() {
                return RefundOrderTask.Parameters.class;
            }

            @Override
            public RefundOrderTask.Parameters convert(RefundOrderTask.Parameters parameters) {
                return parameters;
            }
        }
    }

}
