package ru.yandex.chemodan.app.psbilling.core.billing.users;

import java.math.BigDecimal;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.UUID;

import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.users.OrderDao;
import ru.yandex.chemodan.app.psbilling.core.dao.users.RefundDao;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Refund;
import ru.yandex.chemodan.app.psbilling.core.entities.users.RefundStatus;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPrice;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.users.UserService;
import ru.yandex.chemodan.app.psbilling.core.users.UserServiceManager;
import ru.yandex.chemodan.app.psbilling.core.util.LockService;
import ru.yandex.chemodan.trust.client.TrustClient;
import ru.yandex.chemodan.trust.client.TrustException;
import ru.yandex.chemodan.trust.client.requests.CreateRefundRequest;
import ru.yandex.chemodan.trust.client.requests.OrderRequest;
import ru.yandex.chemodan.trust.client.requests.OrderToRefund;
import ru.yandex.chemodan.trust.client.requests.PaymentRequest;
import ru.yandex.chemodan.trust.client.requests.RefundRequest;
import ru.yandex.chemodan.trust.client.responses.PaymentResponse;
import ru.yandex.chemodan.trust.client.responses.RefundResponse;
import ru.yandex.chemodan.trust.client.responses.SubscriptionResponse;
import ru.yandex.chemodan.trust.client.responses.TrustRefundStatus;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@RequiredArgsConstructor
public class TrustRefundService {
    private static final Logger logger = LoggerFactory.getLogger(TrustRefundService.class);

    private final TrustClient trustClient;
    private final OrderDao orderDao;
    private final RefundDao refundDao;
    private final UserServiceManager userServiceManager;
    private final LockService lockService;
    private final UserProductManager userProductManager;
    private final BazingaTaskManager bazingaTaskManager;
    private final FeatureFlags featureFlags;
    private final BillingActionsReportingService billingActionsReportingService;

    public void scheduleCheckRefund(UUID refundId) {
        bazingaTaskManager.schedule(new CheckRefundTask(refundId));
    }

    public void checkRefund(UUID refundId) {
        Refund refund = refundDao.findById(refundId);
        checkRefund(refund);
    }

    public void checkRefund(Refund refund) {
        Order order = orderDao.findById(refund.getOrderId());

        RefundResponse trustRefundResponse = trustClient.getRefund(RefundRequest.builder()
                .refundId(refund.getTrustRefundId()).trustServiceId(order.getTrustServiceId()).build());
        TrustRefundStatus trustRefundStatus = trustRefundResponse.getRefundStatus();
        logger.info("Refund {} in status {}", refund.getTrustRefundId(), trustRefundStatus);
        switch (trustRefundStatus) {
            case success:
                onSuccessfulRefund(order.getUid(), refund);
                break;
            case error:
                onRefundFailed(refund, order, RefundStatus.ERROR);
                break;
            case failed:
                onRefundFailed(refund, order, RefundStatus.FAILED);
                break;
            case wait_for_notification:
                break;
            case unknown:
            default:
                throw new IllegalStateException(trustRefundResponse.toString());
        }
    }

    public void checkRefund(String trustRefundId) {
        Refund refund =
                refundDao.findByTrustRefundId(trustRefundId).orElseThrow(() -> new NoSuchElementException("Refund " + trustRefundId + " nof found in DB"));
        checkRefund(refund);
    }

    private void onRefundFailed(Refund refund, Order order, RefundStatus refundStatus) {
        UserProductPrice price = userProductManager.findPrice(order.getUserProductPriceId());
        UserProduct product = price.getPeriod().getUserProduct();
        billingActionsReportingService.builder(BillingActionsReportingService.Action.REFUND)
                .status("failed")
                .order(order)
                .userProductPrice(price)
                .refund(refund)
                .userProduct(product)
                .finish();
        refundDao.updateStatus(refund.getId(), refundStatus);
    }

    private void onSuccessfulRefund(String uid, Refund refund) {
        lockService.doWithUserLockedInTransaction(uid, () -> {
            Order order = orderDao.findById(refund.getOrderId());
            UserProductPrice price = userProductManager.findPrice(order.getUserProductPriceId());
            UserProduct product = price.getPeriod().getUserProduct();

            billingActionsReportingService.builder(BillingActionsReportingService.Action.REFUND)
                    .status("success")
                    .order(order)
                    .userProductPrice(price)
                    .refund(refund)
                    .userProduct(product)
                    .finish();
            refundDao.updateStatus(refund.getId(), RefundStatus.SUCCESS);
            disableOrderService(order);
        });
    }

    private void disableOrderService(Order order) {
        if (!order.getUserServiceId().isPresent()) {
            logger.info("order doesn't have service");
            return;
        }

        UserService service = userServiceManager.findById(order.getUserServiceId().get());
        PassportUid passportUid = PassportUid.cons(Long.parseLong(order.getUid()));

        if (service.getTarget() != Target.DISABLED) {
            userServiceManager.disableService(service.getId());
        }
        if (service.getAutoProlongEnabled()) {
            userServiceManager.stopAutoProlong(passportUid, service, Option.empty(), false);
        }

        Option<Order> upgradedOrderO = orderDao.findByUid(passportUid)
                .find(o -> o.getUpgradedOrderIdTo().isNotEmpty()
                        && o.getUpgradedOrderIdTo().get().equals(order.getId()));

        if (upgradedOrderO.isPresent()) {
            logger.info("found upgraded order {}.", upgradedOrderO.get());
            resumeSubscription(upgradedOrderO.get());
        } else {
            logger.debug("upgraded order not found");
        }
    }

    private void resumeSubscription(Order upgradedOrder) {
        Option<Refund> refund = refundDao.getOrderRefunds(upgradedOrder.getId()).firstO();
        if (refund.isNotEmpty()) {
            logger.info("unable to resume order {} cause exist order refund {}", upgradedOrder, refund.get());
            return;
        }

        try {
            trustClient.resumeSubscription(OrderRequest.builder()
                    .uid(PassportUid.cons(Long.parseLong(upgradedOrder.getUid())))
                    .trustServiceId(upgradedOrder.getTrustServiceId())
                    .orderId(upgradedOrder.getTrustOrderId())
                    .build());

            orderDao.onSuccessfulOrderResume(upgradedOrder.getId());
            bazingaTaskManager.schedule(new CheckOrderTask(upgradedOrder.getId(), OrderStatus.INIT));
        } catch (TrustException ex) {
            switch (ex.getStatusCode()) {
                case "achieved_max_qty":
                    logger.info("unable to resume order {} cause max subscription count limit reached", upgradedOrder);
                    return;
                case "subs_not_finished":
                    logger.warn("unable to resume order {} cause subscription is not stopped", upgradedOrder);
                    bazingaTaskManager.schedule(new CheckOrderTask(upgradedOrder.getId(), upgradedOrder.getStatus()));
                    return;
                case "subs_expired":
                    logger.info("unable to resume order {} cause paid period has ended", upgradedOrder);
                    return;
                case "subs_not_started":
                    logger.info("unable to resume order {} cause subscription has not started", upgradedOrder);
                    return;
                default:
                    throw ex;
            }
        }
    }

    public Option<Refund> refundLastOrderPayment(UserService userService, String reason) {
        Validate.isTrue(userService.getLastPaymentOrderId().isPresent());
        Order order = orderDao.findById(userService.getLastPaymentOrderId().get());

        return refundLastOrderPayment(order, reason);
    }

    public Option<Refund> refundLastOrderPayment(Order order, String reason) {
        SubscriptionResponse subscription = trustClient.getSubscription(OrderRequest.builder()
                .uid(PassportUid.cons(Long.parseLong(order.getUid())))
                .trustServiceId(order.getTrustServiceId())
                .orderId(order.getTrustOrderId())
                .build());
        if (subscription.getSubscriptionState().isInTrial() && subscription.getCurrentAmount().orElse(BigDecimal.ZERO).compareTo(BigDecimal.ZERO) == 0) {
            logger.info("subscription {} is in trial state and current amount is 0. " +
                    "No need to refund, just cancel subscription", subscription);
            userServiceManager.stopTrustSubscription(order);
            disableOrderService(order);
            return Option.empty();
        }
        PaymentResponse payment = trustClient.getPayment(PaymentRequest.builder()
                .trustServiceId(order.getTrustServiceId())
                .purchaseToken(subscription.getPaymentIds().last()).build());
        return refundLastOrderPayment(order, payment, reason);
    }

    private Option<Refund> refundLastOrderPayment(Order order, PaymentResponse paymentResponse, String reason) {
        Validate.in(order.getStatus(), Cf.list(OrderStatus.PAID, OrderStatus.UPGRADED));
        PassportUid uid = PassportUid.cons(Long.parseLong(order.getUid()));

        if (doAlreadyExistRefunds(order, paymentResponse)) {
            return Option.empty();
        }
        if (paymentResponse.getPaymentStatus().isRefunded()) {
            throw new IllegalStateException(
                    "Payment already refunded, payment status: " + paymentResponse.getPaymentStatus());
        }
        if (!paymentResponse.getPaymentStatus().isSuccessful()) {
            throw new IllegalStateException("Payment is not refundable: " + paymentResponse.getPaymentStatus());
        }

        String refundId = trustClient.createRefund(CreateRefundRequest.builder()
                .purchaseToken(paymentResponse.getPurchaseToken())
                .trustServiceId(order.getTrustServiceId())
                .reasonDesc(reason)
                .uid(uid)
                .orders(Cf.list(new OrderToRefund(order.getTrustOrderId(), 1)))
                .build());

        Refund refund = refundDao.create(RefundDao.InsertData.builder()
                .orderId(order.getId())
                .trustPaymentId(paymentResponse.getPurchaseToken())
                .trustRefundId(refundId)
                .build());

        trustClient.startRefund(RefundRequest.builder()
                .uid(uid)
                .trustServiceId(order.getTrustServiceId())
                .refundId(refundId)
                .build());

        return Option.of(refund);
    }

    private boolean doAlreadyExistRefunds(Order order, PaymentResponse paymentResponse) {
        ListF<Refund> orderRefunds = refundDao.getOrderRefunds(order.getId())
                .filter(r -> Objects.equals(r.getTrustPaymentId(), paymentResponse.getPurchaseToken()));
        ListF<Refund> refundsInProcess =
                orderRefunds.filter(r -> !RefundStatus.FINAL_STATUSES.containsTs(r.getStatus()));
        if (refundsInProcess.isNotEmpty()) {
            logger.info("Refunds " + refundsInProcess.map(Refund::getTrustPaymentId) +
                    " already in process, in statuses " + refundsInProcess.map(Refund::getStatus));
            return true;
        } else if (orderRefunds.find(r -> r.getStatus() == RefundStatus.SUCCESS).isPresent()) {
            logger.info("Already refunded");
            return true;
        }
        return false;
    }

    public boolean isRefunded(Order order) {
        return refundDao.getOrderRefunds(order.getId()).stream()
                .map(Refund::getStatus)
                .anyMatch(RefundStatus.SUCCESS::equals);

    }
}
