package ru.yandex.chemodan.app.psbilling.core.billing.users;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.NoSuchElementException;
import java.util.UUID;

import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.billing.TrustDeveloperPayloadCreator;
import ru.yandex.chemodan.app.psbilling.core.billing.users.processors.OrderProcessorFacade;
import ru.yandex.chemodan.app.psbilling.core.config.Settings;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.entities.products.BillingType;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Refund;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserInfo;
import ru.yandex.chemodan.app.psbilling.core.products.TrialDefinition;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPeriod;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPrice;
import ru.yandex.chemodan.app.psbilling.core.users.UserInfoService;
import ru.yandex.chemodan.app.psbilling.core.users.UserService;
import ru.yandex.chemodan.app.psbilling.core.users.UserServiceManager;
import ru.yandex.chemodan.app.psbilling.core.util.JsonHelper;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.mpfs.UserBlockedException;
import ru.yandex.chemodan.mpfs.UserNotInitializedException;
import ru.yandex.chemodan.trust.client.TrustClient;
import ru.yandex.chemodan.trust.client.requests.AfsParams;
import ru.yandex.chemodan.trust.client.requests.CreateOrderRequest;
import ru.yandex.chemodan.trust.client.requests.CreatePaymentRequest;
import ru.yandex.chemodan.trust.client.requests.PaymentOrder;
import ru.yandex.chemodan.trust.client.requests.PaymentRequest;
import ru.yandex.chemodan.trust.client.responses.PaymentResponse;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@RequiredArgsConstructor
public class UserBillingService {
    private static final Logger logger = LoggerFactory.getLogger(UserBillingService.class);
    private static final String CARD_PAYMENT_METHOD = "trust_web_page";
    private static final String NON_CLIENT_REQUEST_AFS_PARAM = "mit";

    private final TrustClient trustClient;
    private final UserInfoService userInfoService;
    private final String trustNotificationUrl;
    private final TrustRefundService trustRefundService;
    private final OrderProcessorFacade orderProcessorFacade;
    private final UserServiceManager userServiceManager;
    private final UserProductManager userProductManager;
    private final MpfsClient mpfsClient;
    private final Settings settings;
    private final FeatureFlags featureFlags;
    private final TrustDeveloperPayloadCreator trustDeveloperPayloadCreator;

    public Option<Refund> initRefund(UUID userServiceId, String reason) {
        UserService service = userServiceManager.findById(userServiceId);
        return trustRefundService.refundLastOrderPayment(service, reason);
    }

    public Option<Refund> initRefund(Order order, String reason) {
        return trustRefundService.refundLastOrderPayment(order, reason);
    }

    public void checkRefund(String trustRefundId) {
        trustRefundService.checkRefund(trustRefundId);
    }

    public void checkOrder(String trustOrderId) {
        orderProcessorFacade.processByTrustOrderId(trustOrderId);
    }

    public void scheduleCheckOrder(Order order) {
        orderProcessorFacade.scheduleCheckOrder(order.getId(), order.getStatus());
    }

    public PaymentInfo initPayment(PassportUid uid, String userIp, String domainSuffix,
                                   String returnPath, TrustFormTemplate paymentFormTemplate,
                                   String periodCode, Option<String> language,
                                   Option<String> currency, Boolean disableTrustHeader, Option<String> loginId,
                                   boolean useTemplate) {
        return initPayment(uid, userIp, domainSuffix, Option.of(returnPath), Option.of(paymentFormTemplate),
                periodCode, language, currency,
                disableTrustHeader, loginId, CARD_PAYMENT_METHOD, true, false, useTemplate);
    }

    public PaymentInfo initPayment(PassportUid uid, String userIp, String domainSuffix,
                                   Option<String> returnPath, Option<TrustFormTemplate> paymentFormTemplate,
                                   String periodCode, Option<String> language,
                                   Option<String> currency, Boolean disableTrustHeader, Option<String> loginId,
                                   String paymentMethod, boolean validatePrice, boolean isAutomaticPayment,
                                   boolean useTemplate) {
        UserProductPeriod period = userProductManager.findPeriodByCode(periodCode);
        UserProduct userProduct = userProductManager.findById(period.getUserProductId());
        Option<String> region = userInfoService.findOrCreateUserInfo(uid).getRegionId();
        Option<String> conflictProductCurrency = userServiceManager.findEnabledConflicting(uid.toString(),
                        userProduct.getId())
                .filter(us -> us.getPrice().isPresent())
                .firstO()
                .flatMapO(UserService::getPrice)
                .map(UserProductPrice::getCurrencyCode);
        Option<String> paymentCurrency;
        if (conflictProductCurrency.isPresent()) {
            logger.info("Bought product exists, use bought product currency {}", conflictProductCurrency);
            paymentCurrency = conflictProductCurrency;
        } else {
            paymentCurrency = currency;
        }

        UserProductPrice price = period.selectPriceForRegion(settings, region, paymentCurrency)
                .orElseThrow(() -> new NoSuchElementException("Unable to determine price for " + uid));
        Validate.equals(userProduct.getBillingType(), BillingType.TRUST);
        if (validatePrice) {
            userProductManager.validatePrice(uid, price);
        }

        UserInfo userInfo = userInfoService.findOrCreateUserInfo(uid);

        Integer trustServiceId =
                userProduct.getTrustServiceId().orElseThrow(() -> new NoSuchElementException("trust service id not " +
                        "defined for " + userProduct));
        String newTrustOrderId = UUID.randomUUID().toString();
        CreateOrderRequest.CreateOrderRequestBuilder createOrderRequestBuilder = CreateOrderRequest.builder()
                .orderId(newTrustOrderId)
                .trustServiceId(trustServiceId)
                .productId(period.getTrustProductId(true))
                .regionId(userInfo.getRegionId().orElse((String) null))
                .userIp(userIp)
                .uid(uid)
                .developerPayload(trustDeveloperPayloadCreator.createCreateOrderPayload(period, language, disableTrustHeader));

        String trustOrderId = trustClient.createSubscription(createOrderRequestBuilder.build());

        //проверка после факапа с временными адресами https://st.yandex-team.ru/SPI-35908
        //https://st.yandex-team.ru/CHEMODAN-82235
        //траст вернул не тот orderId который мы ему прислали, вероятно у нас такой уже есть, проверяем
        boolean orderIsNew = newTrustOrderId.equals(trustOrderId);

        Order order;
        // CHEMODAN-82785 - Удалить getSubscriptionOrderUseCreateIfNotExists как только обкатаем функционал.
        if (featureFlags.getSubscriptionOrderUseCreateIfNotExists().isEnabled()) {
            // Новый функционал
            order = orderProcessorFacade.createIfNotExistsSubscriptionOrder(uid, trustOrderId, price);
            if (featureFlags.getCheckTrustOrderId().isEnabled() && !orderIsNew) {
                //по этому заказу уже была успешная покупка выходим
                checkUnavailablePrice(order, price, uid);
            }
        } else {
            // Старый функционал, удалить после прокатки
            if (featureFlags.getCheckTrustOrderId().isEnabled() && !orderIsNew) {
                //по этому заказу уже была успешная покупка выходим
                Option<Order> paidOrder = orderProcessorFacade.findOrder(trustOrderId);
                if (paidOrder.isPresent()) {
                    checkUnavailablePrice(paidOrder.get(), price, uid);
                }

            }
            order = orderProcessorFacade.createSubscriptionOrder(uid, trustOrderId, price);
        }

        logger.debug("Order {}", order);

        String purchaseToken = trustClient.createPayment(CreatePaymentRequest.builder()
                .domainSuffix(domainSuffix)
                .lang(language.orElse((String) null))
                .trustServiceId(trustServiceId)
                .notificationUrl(trustNotificationUrl.replaceAll("!ORDER_ID!", trustOrderId))
                .regionId(userInfo.getRegionId().orElse((String) null))
                .returnPath(returnPath.map(path -> path.replaceAll("!ORDER_ID!", order.getId().toString())).orElse((String) null))
                .currency(price.getCurrencyCode())
                .uid(uid)
                .userIp(userIp)
                .templateTag(paymentFormTemplate.map(TrustFormTemplate::getTag).orElse((String) null))
                .paymentMethod(paymentMethod)
                .orders(Collections.singletonList(new PaymentOrder(trustOrderId, 1)))
                .developerPayload(JsonHelper.asString(trustDeveloperPayloadCreator.createCreatePaymentPayload(useTemplate, userProduct)))
                .afsParams(AfsParams.builder()
                        .loginId(loginId.orElse((String) null))
                        .request(isAutomaticPayment ? NON_CLIENT_REQUEST_AFS_PARAM : null)
                        .build())
                .build());

        PaymentResponse paymentResponse = trustClient.startPayment(PaymentRequest.builder()
                .regionId(userInfo.getRegionId().orElse((String) null))
                .uid(uid)
                .userIp(userIp)
                .trustServiceId(trustServiceId)
                .purchaseToken(purchaseToken)
                .build());
        validatePrice(paymentResponse, userProduct);

        //проверка после факапа с временными адресами https://st.yandex-team.ru/SPI-35908
        //https://st.yandex-team.ru/CHEMODAN-82235
        //  проверяем что за время пути наш заказ не успел оплатиться
        if (featureFlags.getCheckTrustOrderId().isEnabled() && !orderIsNew) {
            Option<Order> orderO = orderProcessorFacade.findOrder(trustOrderId);
            if (orderO.isPresent()) {
                Order existingOrder;
                //double check if not paid
                logger.debug("Double check if not paid. Order {}", orderO);
                if (!OrderStatus.TERMINAL_STATUSES.contains(orderO.get().getStatus())) {
                    checkOrder(trustOrderId);
                    existingOrder = orderProcessorFacade.findOrder(trustOrderId).get();
                    logger.debug("Double check if not paid. New existing Order {}", existingOrder);
                } else {
                    existingOrder = orderO.get();
                }

                checkUnavailablePrice(existingOrder, price, uid);
            }
        }

        return new PaymentInfo(order, paymentResponse.getPaymentUrl());
    }

    public UUID createFreeOrder(PassportUid uid, String periodCode) {
        logger.info("Requested to create a free order for UID {} and period {}", uid, periodCode);
        UserProductPeriod period = userProductManager.findPeriodByCode(periodCode);
        UserProduct userProduct = period.getUserProduct();
        Option<String> region = userInfoService.findOrCreateUserInfo(uid).getRegionId();
        UserProductPrice price = period.selectPriceForRegion(settings, region, Option.empty())
                .orElseThrow(() -> new NoSuchElementException("Unable to determine price for " + uid + " and region " + region));
        if (!BillingType.FREE.equals(userProduct.getBillingType())) {
            throw new IllegalArgumentException("product is not free: " + periodCode);
        }

        String newTrustOrderId = "free_" + UUID.randomUUID();
        Order order = orderProcessorFacade.createIfNotExistsOrderOfTypePromocodeOrder(uid, newTrustOrderId, price);
        orderProcessorFacade.processOrder(order);
        logger.debug("Created free order {}", order);
        return order.getId();
    }

    private void checkUnavailablePrice(Order order, UserProductPrice price, PassportUid uid) {
        if (OrderStatus.TERMINAL_STATUSES.contains(order.getStatus())) {
            throw new A3ExceptionWithStatus(
                    "price_unavailable",
                    "the price " + price + " doesn't available for " + uid,
                    HttpStatus.SC_400_BAD_REQUEST);
        }
    }

    private void validatePrice(PaymentResponse paymentResponse, UserProduct userProduct) {
        Option<TrialDefinition> trialO = userProduct.getTrialDefinition();
        if (!trialO.isPresent()) {
            return;
        }
        TrialDefinition trial = trialO.get();
        if (trial.getPrice().compareTo(BigDecimal.ZERO) != 0) {
            return;
        }

        if (BigDecimal.ZERO.compareTo(paymentResponse.getAmount()) != 0) {
            throw new IllegalStateException("created payment with not zero amount for trial product");
        }
    }

    public boolean isDiskProEnable(PassportUid uid) {
        return RetryUtils.retry(logger, 3, 100, 2, () -> {
            try {
                return mpfsClient.getFeatureToggles(MpfsUser.of(uid))
                        .getOrDefault("disk_pro", false);
            } catch (UserNotInitializedException | PermanentHttpFailureException | UserBlockedException e) {
                logger.info("mpfs error", e);
                return false;
            }
        });
    }
}
