package ru.yandex.chemodan.app.psbilling.core.billing.users.processors;

import java.util.UUID;

import lombok.AllArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.billing.users.BillingActionsReportingService;
import ru.yandex.chemodan.app.psbilling.core.config.Settings;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.users.OrderDao;
import ru.yandex.chemodan.app.psbilling.core.dao.users.UserServiceDao;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserServiceBillingStatus;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPrice;
import ru.yandex.chemodan.app.psbilling.core.promos.PromoService;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.users.UserService;
import ru.yandex.chemodan.app.psbilling.core.users.UserServiceManager;
import ru.yandex.chemodan.app.psbilling.core.util.LockService;
import ru.yandex.chemodan.trust.client.TrustClient;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;


@AllArgsConstructor
public abstract class AbstractSubscriptionProcessor {
    private static final Logger logger = LoggerFactory.getLogger(AbstractSubscriptionProcessor.class);
    protected final BazingaTaskManager bazingaTaskManager;
    protected final TrustClient trustClient;
    protected final OrderDao orderDao;
    protected final UserServiceManager userServiceManager;
    protected final LockService lockService;
    protected final UserProductManager userProductManager;
    protected final TaskScheduler taskScheduler;
    protected final UserServiceDao userServiceDao;
    protected final PromoService promoService;
    protected final Settings settings;
    protected final FeatureFlags featureFlags;
    protected final BillingActionsReportingService billingActionsReportingService;

    protected void prolongServiceWithoutLocking(
            Order order, UserProductPrice price, UserService userService, Instant subscribedUntil,
            int subscriptionsCount, UserServiceBillingStatus serviceBillingStatus) {

        // если подписка просрочилась уже на определенное время, и траст ее не продлевает, то отключаем ее
        if (subscribedUntil.plus(settings.getAcceptableUserServiceCheckDateExpirationTime()).isBeforeNow()) {
            logger.warn("service {} for order {} has subscribed until = {} (expired for more than {} hours). " +
                            "Service will be disabled", userService.getId(), order.getId(), subscribedUntil,
                    settings.getAcceptableUserServiceCheckDateExpirationTime());
            disableUserService(userService, subscribedUntil, true);
            return;
        }

        logger.info("updating next check date for {}, subscribed until: {}", userService, subscribedUntil);
        Option<Instant> oldExpirationDate = userService.getNextCheckDate();
        userServiceManager.updateNextCheckDate(userService.getId(), subscribedUntil, serviceBillingStatus);
        UserProduct userProduct = userService.getUserProduct();

        if (order.getSubscriptionsCount() != subscriptionsCount ||
                (oldExpirationDate.isPresent() && oldExpirationDate.get().plus(Duration.standardMinutes(1))
                        .isBefore(subscribedUntil))
        ) { //for inapps there is no subscriptionsCount, so we consider prolong when next check date is moved
            // to future

            CalculatedPrice priceValue = getPriceForPeriod(price, order);
            billingActionsReportingService.builder(BillingActionsReportingService.Action.PROLONG_AUTO)
                    .status("success")
                    .order(order)
                    .userProductPrice(price)
                    .userProduct(userProduct)
                    .serviceBillingStatus(serviceBillingStatus)
                    .oldExpirationDate(oldExpirationDate.orElse((Instant) null))
                    .newExpirationDate(subscribedUntil)
                    .calculatedPrice(priceValue)
                    .finish();
            orderDao.updateSubscriptionsCount(order.getId(), subscriptionsCount);
        }
    }

    protected CalculatedPrice getPriceForPeriod(UserProductPrice price, Order order) {
        return new CalculatedPrice(price.getPrice());
    }



    @AllArgsConstructor
    public static class UserServiceCreatedResult {
        UserService userService;
        Option<String> usedPromoTemplateCode;
    }
    protected UserServiceCreatedResult createUserServiceForOrder(
            Order order, Instant subscribedUntil, UserServiceBillingStatus serviceBillingStatus,
            int subscriptionsCount) {
        return createUserServiceForOrder(order, subscribedUntil, serviceBillingStatus, subscriptionsCount, true);
    }

    protected UserServiceCreatedResult createUserServiceForOrder(
            Order order, Instant subscribedUntil, UserServiceBillingStatus serviceBillingStatus,
            int subscriptionsCount, boolean checkForConflicts) {
        if (checkForConflicts) {
            checkExistConflictingProducts(order.getUid(), order.getUserProductPriceId());
        }
        UserService userService = userServiceManager.createUserService(order, subscribedUntil, serviceBillingStatus);
        orderDao.onSuccessfulOrderPurchase(order.getId(), Option.of(userService.getId()), subscriptionsCount);
        Option<String> usedPromoTemplateCode = promoService.setPromoUsedState(userProductManager,
                PassportUid.cons(Long.parseLong(order.getUid())),
                order.getUserProductPriceId());
        return new UserServiceCreatedResult(userService, usedPromoTemplateCode);
    }

    private void checkExistConflictingProducts(String uid, UUID priceId) {
        UserProductPrice price = userProductManager.findPrice(priceId);
        UserProduct userProduct = price.getPeriod().getUserProduct();
        ListF<UserService> activeServices = userServiceManager.findEnabled(uid, Option.empty());
        ListF<UserService> conflictingService =
                activeServices.filter(s -> s.getUserProduct().conflictsWith(userProduct.getId()));
        if (!conflictingService.isEmpty()) {
            throw new IllegalStateException("Unable to create new service of product " + userProduct.getCode() +
                    " for user " + uid + ", due to exist conflicting services " + conflictingService.map(UserService::getId));
        }
    }

    protected void processStopSubscription(Order orderUnlocked, Instant subscribedUntil,
                                           int lastSubscriptionsCount) {
        PassportUid uid = PassportUid.cons(Long.parseLong(orderUnlocked.getUid()));
        Option<UUID> disabledService = lockService.doWithUserLockedInTransaction(uid.toString(),
                () -> {
                    Order order = orderDao.findById(orderUnlocked.getId());
                    if (order.getStatus() == OrderStatus.ON_HOLD) {
                        orderDao.unHoldOrder(order.getId());
                    }
                    return stopSubscriptionWithoutLock(
                            order, subscribedUntil, false, lastSubscriptionsCount);
                });
        if (orderUnlocked.getStatus() == OrderStatus.ON_HOLD &&
                featureFlags.getLongBillingPeriodForTrustEmailEnabled().isEnabledForUid(uid)) {
            taskScheduler.scheduleHoldIsOverEmailTask(uid, orderUnlocked.getUserServiceId().get(),
                    orderUnlocked.getUserProductPriceId());
            return;
        }
        disabledService.ifPresent(userServiceId -> taskScheduler.scheduleSubscriptionFinishedEmailTask(
                uid,
                userServiceId,
                orderUnlocked.getType(),
                orderUnlocked.getUserProductPriceId()
        ));
    }


    protected Option<UUID> stopSubscriptionWithoutLock(Order order, Instant subscribedUntil,
                                                       boolean forceDisableService,
                                                       int lastSubscriptionsCount) {
        Option<UserService> userServiceO = order.getUserServiceId().map(userServiceManager::findById);

        if (!userServiceO.isPresent()) {
            if (subscribedUntil.isBeforeNow() || forceDisableService) {
                logger.warn("subscription ended and stopped for order without user service. order: {}",
                        order);
                return Option.empty();
            } else {
                //по какой-то причине пришел ордер, который отменили. такое может быть если прилетел ордер,
                // который был в престейбле. его там отменили, но потом попробовали купить в проде.
                // для однообразности логики, сделаем что сказал траст - выдадим услугу по указанную в
                // ордере дату
                logger.info("received already stopped for autoprolong order, creating service");
                UserServiceCreatedResult serviceReport = createUserServiceForOrder(order, subscribedUntil,
                        UserServiceBillingStatus.PAID, lastSubscriptionsCount);
                userServiceO = Option.of(serviceReport.userService);
            }
        }

        UserService service = userServiceO.get();
        boolean disabled = disableUserService(service, subscribedUntil, forceDisableService);
        return disabled ? Option.of(service.getId()) : Option.empty();
    }

    private boolean disableUserService(UserService service, Instant subscribedUntil, boolean forceDisableService) {
        if (subscribedUntil.isAfterNow() && service.getAutoProlongEnabledO().orElse(false)) {
            userServiceManager.stopAutoProlong(service.getId(), subscribedUntil);
            logger.info("Stopping autoprolong for {}, subscribed until: {}", service.getId(),
                    subscribedUntil);
        }
        if (service.getTarget() == Target.DISABLED) {
            logger.info("Service {} already stopped", service);
            return false;
        } else if (subscribedUntil.isBeforeNow() || forceDisableService) {
            logger.info("Disabling service {}", service);
            userServiceManager.disableService(service.getId());
            return true;
        } else {
            return false;
        }
    }

    protected MailContext buildGraceEmailContext(UUID userServiceId, String uid) {
        return MailContext.builder()
                .to(PassportUid.cons(Long.parseLong(uid)))
                .userServiceId(Option.of(userServiceId.toString()))
                .build();
    }

    void processHoldOrder(Order orderUnlocked, Instant subscriptionUntil, int subscriptionPeriodCount) {
        OrderStatus currentStatus = orderUnlocked.getStatus();
        //отключаем подписку независимо от даты ее окончания, когда расхолдят - будем заново выдавать
        lockService.doWithUserLockedInTransaction(orderUnlocked.getUid(),
                () -> {
                    Order order = orderDao.findById(orderUnlocked.getId());
                    stopSubscriptionWithoutLock(order, subscriptionUntil, true,
                            subscriptionPeriodCount);
                    orderDao.holdOrder(order.getId());
                });
        //если до того как мы сюда пришли подписка не была в холде - значит переход произошел сейчас. Отправляем письмо
        if (currentStatus != OrderStatus.ON_HOLD) {
            taskScheduler.scheduleOnHoldEmailTask(orderUnlocked);
        }
    }
}
