package ru.yandex.chemodan.app.psbilling.core.billing.users.processors;

import java.util.NoSuchElementException;
import java.util.Objects;

import lombok.SneakyThrows;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.billing.users.BillingActionsReportingService;
import ru.yandex.chemodan.app.psbilling.core.config.Settings;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.users.OrderDao;
import ru.yandex.chemodan.app.psbilling.core.dao.users.UserServiceDao;
import ru.yandex.chemodan.app.psbilling.core.entities.InappStore;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderType;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserServiceBillingStatus;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPeriod;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPrice;
import ru.yandex.chemodan.app.psbilling.core.promos.PromoService;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.users.UserService;
import ru.yandex.chemodan.app.psbilling.core.users.UserServiceManager;
import ru.yandex.chemodan.app.psbilling.core.util.LockService;
import ru.yandex.chemodan.trust.client.TrustClient;
import ru.yandex.chemodan.trust.client.TrustException;
import ru.yandex.chemodan.trust.client.requests.OrderRequest;
import ru.yandex.chemodan.trust.client.requests.SendInappReceiptRequest;
import ru.yandex.chemodan.trust.client.responses.AppleReceiptResponse;
import ru.yandex.chemodan.trust.client.responses.InappSubscription;
import ru.yandex.chemodan.trust.client.responses.InappSubscriptionResponse;
import ru.yandex.chemodan.trust.client.responses.InappSubscriptionState;
import ru.yandex.chemodan.trust.client.responses.ProcessInappReceiptResponse;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class InappSubscriptionProcessor extends AbstractSubscriptionProcessor implements OrderProcessor {
    private static final Logger logger = LoggerFactory.getLogger(InappSubscriptionProcessor.class);

    private final Integer inappTrustServiceId;
    private final TrustInappExceptionHandler trustExceptionHandler;

    public InappSubscriptionProcessor(BazingaTaskManager bazingaTaskManager, TrustClient trustClient, OrderDao orderDao,
                                      UserServiceManager userServiceManager, LockService lockService,
                                      UserProductManager userProductManager, Integer inappTrustServiceId,
                                      TaskScheduler taskScheduler, UserServiceDao userServiceDao,
                                      PromoService promoService, Settings settings, FeatureFlags featureFlags,
                                      BillingActionsReportingService billingActionsReportingService) {
        super(bazingaTaskManager, trustClient, orderDao, userServiceManager, lockService,
                userProductManager, taskScheduler, userServiceDao, promoService, settings, featureFlags,
                billingActionsReportingService);
        this.inappTrustServiceId = inappTrustServiceId;
        this.trustExceptionHandler = new TrustInappExceptionHandler(settings, lockService, this::forceDisableService);
    }

    @Override
    @SneakyThrows
    public void processOrder(Order order) {
        PassportUid uid = PassportUid.cons(Long.parseLong(order.getUid()));
        InappSubscriptionResponse trustResponse = trustClient.getInappSubscription(OrderRequest.builder()
                .uid(uid).orderId(order.getTrustOrderId())
                .trustServiceId(order.getTrustServiceId())
                .build());
        InappSubscription subscription;

        if (trustResponse.getSubscription().getSyncTime().isBefore(Instant.now().minus(settings.getAcceptableNotSyncedTimeInHours()))) {
            InappSubscription staleSubscription = trustResponse.getSubscription();
            try {
                logger.info("subscription sync time is too old. Resyncing...");
                resyncSubscription(order, staleSubscription);
            } catch (TrustException ex) {
                trustExceptionHandler.handleTrustException(ex, staleSubscription, order, true);
                return;
            }

            InappSubscriptionResponse resyncedResponse = trustClient.getInappSubscription(OrderRequest.builder()
                    .uid(uid).orderId(order.getTrustOrderId())
                    .trustServiceId(order.getTrustServiceId())
                    .build());
            subscription = resyncedResponse.getSubscription();
            if (staleSubscription.getState() != subscription.getState()) {
                logger.info("subscription state changed after resyncing from {} to {}",
                        staleSubscription.getState(), subscription.getState());
            }
            if (subscription.getSyncTime() == staleSubscription.getSyncTime()) {
                logger.warn("sync time has not changed after resyncing {}", resyncedResponse);
            }
        } else {
            subscription = trustResponse.getSubscription();
        }

        syncOrderAndServices(order, uid, subscription);
    }

    public Order processInappTrustSubscription(InappSubscription inappTrustSubscription, PassportUid uid,
                                               String packageName, Option<Order> expectedExistOrder) throws UnknownProductException {
        Option<Order> order = findTrustOrder(inappTrustSubscription.getSubscriptionId());
        // optimistic lock check CHEMODAN-81425
        checkOrderNotChanged(expectedExistOrder, order);

        if (order.isEmpty()) {
            order = Option.of(createInappOrder(inappTrustSubscription, packageName, OrderStatus.PAID));
        }
        syncOrderAndServices(order.get(), uid, inappTrustSubscription);
        return orderDao.findById(order.get().getId());
    }

    public Option<Order> findTrustOrder(String trustOrderId) {
        return orderDao.findByTrustOrderId(trustOrderId);
    }

    public Order createInappOrder(InappSubscription subscription, String packageName, OrderStatus status) throws
            UnknownProductException {
        UserProductPrice price = getInappSubscriptionPrice(subscription);

        return orderDao.createOrUpdate(OrderDao.InsertData.builder()
                .trustOrderId(subscription.getSubscriptionId())
                .type(OrderType.INAPP_SUBSCRIPTION)
                .uid(subscription.getUid())
                .userProductPriceId(price.getId())
                .trustServiceId(price.getPeriod().getUserProduct().getTrustServiceId()
                        .orElseThrow(() -> new NoSuchElementException("Not found trust service id for period with " +
                                "trust code " + subscription.getProductId())))
                .packageName(Option.ofNullable(packageName))
                .status(Option.of(status))
                .build());
    }

    public ProcessInappReceiptResponse checkInappReceipt(PassportUid uid, InappStore inappStoreType,
                                                         String currency, String receipt) {
        try {
            return trustClient.processInappReceipt(SendInappReceiptRequest.builder()
                    .currency(currency)
                    .receipt(receipt)
                    .uid(uid)
                    .storeId(inappStoreType.toTrustType())
                    .trustServiceId(inappTrustServiceId)
                    .build());
        } catch (Exception e) {
            logger.error("Failed to process receipt in trust", e);
            throw new A3ExceptionWithStatus("invalid-receipt", HttpStatus.SC_400_BAD_REQUEST);
        }
    }

    public AppleReceiptResponse checkAppstoreReceipt(String receipt) {
        try {
            return trustClient.checkAppstoreReceipt(receipt, inappTrustServiceId);
        } catch (TrustException e) {
            logger.error("Failed to receive raw receipt", e);
            throw new A3ExceptionWithStatus("invalid-receipt", HttpStatus.SC_400_BAD_REQUEST);
        }
    }

    protected void createOrProlongServiceForOrder(
            Order orderUnlocked, UserProductPrice price, Instant subscribedUntil,
            UserServiceBillingStatus serviceBillingStatus) {
        lockService.doWithUserLockedInTransaction(orderUnlocked.getUid(), () -> {
            Order order = orderDao.findById(orderUnlocked.getId());
            logger.info("order from db: {}", order);
            UserProduct product = price.getPeriod().getUserProduct();
            if (!Objects.equals(order.getUserProductPriceId(), price.getId())) {
                logger.info("price for order {} changed to {}, order will be upgraded", order, price);
                updateOrderPrice(order, price, subscribedUntil, serviceBillingStatus);
                billingActionsReportingService.builder(BillingActionsReportingService.Action.ORDER_UPGRADED)
                        .status("success")
                        .order(order)
                        .userProductPrice(price)
                        .userProduct(product)
                        .finish();
                return;
            }

            Option<UserService> serviceO = order.getUserServiceId().map(userServiceManager::findById);
            if (serviceO.map(UserService::getTarget).containsTs(Target.ENABLED)) {
                UserService userService = serviceO.get();
                prolongServiceWithoutLocking(order, price, userService, subscribedUntil, -1, serviceBillingStatus);
            } else {
                boolean isOnHold = order.getStatus() == OrderStatus.ON_HOLD;
                createUserServiceForOrder(order, subscribedUntil, serviceBillingStatus, -1,
                        !isOnHold);
                BillingActionsReportingService.Action action;
                if (isOnHold) {
                    action = BillingActionsReportingService.Action.RESTORE_FROM_HOLD;
                } else {
                    action = BillingActionsReportingService.Action.BUY_NEW;
                }
                billingActionsReportingService.builder(action)
                        .status("success")
                        .order(order)
                        .userProductPrice(price)
                        .userProduct(product)
                        .newExpirationDate(subscribedUntil)
                        .finish();
            }
        });
    }

    private void syncOrderAndServices(Order orderOld, PassportUid uid, InappSubscription subscription) throws UnknownProductException {
        syncOrderAndServices(orderOld, uid, subscription, subscription.getState());
    }

    private void syncOrderAndServices(Order order, PassportUid uid, InappSubscription subscription,
                                      InappSubscriptionState state) throws UnknownProductException {
        if (!Objects.equals(order.getInappSynchronizationDate().orElse((Instant) null), (subscription.getSyncTime()))) {
            order = orderDao.updateInappSyncDate(order.getId(), subscription.getSyncTime());
        }
        Order finalOrder = order;
        lockService.doWithOrderLockedInTransaction(order.getId(), lockedOrder -> {
            // optimistic lock check CHEMODAN-81425
            checkOrderNotChanged(Option.of(finalOrder), Option.of(lockedOrder));
            logger.info("Before sync: order {}, subscription: {}", finalOrder, subscription);
            switch (state) {
                case ACTIVE:
                    onSubscriptionActive(finalOrder, uid, subscription);
                    break;
                case FINISHED:
                    onSubscriptionCancelled(finalOrder, subscription);
                    break;
                case ON_HOLD:
                    onSubscriptionOnHold(finalOrder, subscription);
                    break;
                case IN_GRACE:
                    onSubscriptionInGrace(finalOrder, uid, subscription);
                    break;
                case NOT_SYNCHRONIZED:
                    onOrderNotSynced(finalOrder, subscription);
                    break;
                default:
                    throw new IllegalStateException("unknown inapp subscription state");
            }
        });
    }

    private void onOrderNotSynced(Order orderOld, InappSubscription subscription) {
        logger.info("order {} not synced in trust, response: {}", orderOld, subscription);
        if (subscription.getSubscriptionUntil().plus(Duration.standardMinutes(1)).isBeforeNow()) {
            try {
                resyncSubscription(orderOld, subscription);
            } catch (TrustException ex) {
                trustExceptionHandler.handleTrustException(ex, subscription, orderOld, false);
            }
        }
    }

    private void onSubscriptionActive(Order order, PassportUid uid, InappSubscription subscription) throws UnknownProductException {
        logger.info("onInappSubscriptionActive: {} {}", order, subscription);
        // it's possible situation when product in trust changed due to product upgrade in store
        createOrProlongServiceOrOrder(order, uid, subscription, UserServiceBillingStatus.PAID);
    }

    private void onSubscriptionInGrace(Order order, PassportUid uid, InappSubscription subscription) throws UnknownProductException {
        logger.info("onInappSubscriptionInGrace: {} {}", order, subscription);
        Option<UserServiceBillingStatus> billingStatus = getUserServiceBillingStatus(order);
        createOrProlongServiceOrOrder(order, uid, subscription, UserServiceBillingStatus.WAIT_AUTO_PROLONG);
        if (billingStatus.orElse(UserServiceBillingStatus.PAID) != UserServiceBillingStatus.WAIT_AUTO_PROLONG) {
            //refresh order in case of userService change while processing
            taskScheduler.scheduleInGraceEmailTask(orderDao.findById(order.getId()));
        }
    }

    private Option<UserServiceBillingStatus> getUserServiceBillingStatus(Order order) {
        if (order.getUserServiceId().isEmpty())
            return Option.empty();
        return userServiceManager.findById(order.getUserServiceId().get()).getBillingStatus();
    }

    private void onSubscriptionCancelled(Order order, InappSubscription subscription) {
        logger.info("onInappSubscriptionCancelled: {} {}", order, subscription);
        processStopSubscription(order, subscription.getSubscriptionUntil(), -1);
    }

    private void onSubscriptionOnHold(Order order, InappSubscription subscription) {
        logger.info("HOLD order {}, subscription: {}", order, subscription);
        processHoldOrder(order, subscription.getSubscriptionUntil(), -1);
    }

    private void forceDisableService(Order order, InappSubscription subscription) {
        stopSubscriptionWithoutLock(
                order, subscription.getSubscriptionUntil(), true, -1);
    }

    private void updateOrderPrice(Order order, UserProductPrice price, Instant subscribedUntil,
                                  UserServiceBillingStatus serviceBillingStatus) {
        order.getUserServiceId().ifPresent(userServiceManager::disableService);
        order = orderDao.updateOrderPrice(order.getId(), price.getId(), -1);
        createUserServiceForOrder(order, subscribedUntil, serviceBillingStatus, -1);
    }


    private void createOrProlongServiceOrOrder(Order order, PassportUid uid, InappSubscription subscription,
                                               UserServiceBillingStatus serviceBillingStatus) throws UnknownProductException {
        UserProductPrice price = getInappSubscriptionPrice(subscription);
        if (!order.getUserServiceId().map(userServiceManager::findById).exists(s -> s.getTarget() != Target.DISABLED)) {
            // все сервисы отключены, поэтому у нас режим создания заказа, поэтому создавать его будем от имени того,
            // кто пришел к нам c чеком
            logger.info("Changing uid of order {} to {}", order, uid);
            order = orderDao.changeOrderUid(order.getId(), uid.toString());
        }

        createOrProlongServiceForOrder(order, price, subscription.getSubscriptionUntil(), serviceBillingStatus);
    }

    private UserProductPrice getInappSubscriptionPrice(InappSubscription subscription) throws UnknownProductException {
        // it's possible situation when product in trust changed due to product upgrade in store
        Option<UserProductPeriod> period = userProductManager.findPeriodByCodeO(subscription.getProductId());
        // for inapp always one fake price. Validated here ru.yandex.chemodan.app.psbilling.core.db.PsBillingDatabaseValidationTest.singlePriceForInapp
        return period.orElseThrow(() -> new UnknownProductException(subscription.getProductId())).getPrices().single();
    }

    private void resyncSubscription(Order orderOld, InappSubscription subscription) {
        trustClient.resyncInappSubscription(OrderRequest.builder()
                .uid(PassportUid.cons(Long.parseLong(orderOld.getUid())))
                .orderId(subscription.getSubscriptionId())
                .trustServiceId(orderOld.getTrustServiceId())
                .build());
    }

    private void checkOrderNotChanged(Option<Order> was, Option<Order> now) {
        if (!was.equals(now)) {
            logger.error("order changed during request execution. was {} but now {}", was, now);
            throw new A3ExceptionWithStatus("order changed during request execution", 409);
        }
    }
}
