package ru.yandex.chemodan.app.psbilling.core.billing.users.processors;

import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.UUID;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.billing.users.CheckOrderTask;
import ru.yandex.chemodan.app.psbilling.core.billing.users.CheckUserServiceTask;
import ru.yandex.chemodan.app.psbilling.core.dao.users.OrderDao;
import ru.yandex.chemodan.app.psbilling.core.entities.InappStore;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderType;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserServiceEntity;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPrice;
import ru.yandex.chemodan.trust.client.responses.AppleReceiptResponse;
import ru.yandex.chemodan.trust.client.responses.InappSubscription;
import ru.yandex.chemodan.trust.client.responses.ProcessInappReceiptResponse;
import ru.yandex.chemodan.util.exception.NotFoundException;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class OrderProcessorFacade implements OrderProcessor {
    private final Logger logger = LoggerFactory.getLogger(OrderProcessorFacade.class);

    private final BazingaTaskManager bazingaTaskManager;
    private final TrustSubscriptionProcessor trustSubscriptionProcessor;
    private final InappSubscriptionProcessor inappSubscriptionProcessor;
    private final PromocodeOrderProcessor promocodeOrderProcessor;
    private final OrderDao orderDao;

    public OrderProcessorFacade(BazingaTaskManager bazingaTaskManager, OrderDao orderDao,
                                TrustSubscriptionProcessor trustSubscriptionProcessor,
                                InappSubscriptionProcessor inappSubscriptionProcessor,
                                PromocodeOrderProcessor promocodeOrderProcessor) {
        this.bazingaTaskManager = bazingaTaskManager;
        this.trustSubscriptionProcessor = trustSubscriptionProcessor;
        this.inappSubscriptionProcessor = inappSubscriptionProcessor;
        this.promocodeOrderProcessor = promocodeOrderProcessor;
        this.orderDao = orderDao;
    }

    public FullJobId scheduleCheckOrder(UUID orderId, OrderStatus orderStatus) {
        return bazingaTaskManager.schedule(new CheckOrderTask(orderId, orderStatus));
    }

    public FullJobId scheduleCheckService(UserServiceEntity userService) {
        return bazingaTaskManager.schedule(new CheckUserServiceTask(userService.getId()));
    }

    @Override
    public void processOrder(Order order) {
        getProcessor(order).processOrder(order);
    }

    public Option<Order> findOrder(String trustOrderId) {
        return inappSubscriptionProcessor.findTrustOrder(trustOrderId);
    }

    public Order processInappSubscription(InappSubscription inappTrustSubscription, PassportUid uid, String packageName, Option<Order> expectedExistOrder)
            throws UnknownProductException {
        return inappSubscriptionProcessor.processInappTrustSubscription(inappTrustSubscription, uid, packageName, expectedExistOrder);
    }

    public Order createIfNotExistsSubscriptionOrder(PassportUid uid, String trustOrderId, UserProductPrice price) {
        logger.debug("createIfNotExistsSubscriptionOrder uid=[{}] trustOrderId=[{}] price=[{}]", uid, trustOrderId, price);
        return orderDao.createIfNotExists(buildOrderInsertData(trustOrderId, uid, price));
    }

    public Order createIfNotExistsOrderOfTypePromocodeOrder(PassportUid uid, String trustOrderId, UserProductPrice price) {
        logger.debug("createIfNotExistsOrderOfTypeFreeOrder uid=[{}] trustOrderId=[{}] price=[{}]", uid, trustOrderId, price);
        OrderDao.InsertData.InsertDataBuilder builder = OrderDao.InsertData.builder()
                .trustOrderId(trustOrderId)
                .type(OrderType.PROMOCODE_ORDER)
                .uid(uid.toString())
                .userProductPriceId(price.getId());
        price.getPeriod().getUserProduct().getTrustServiceId().ifPresent(builder::trustServiceId);

        return orderDao.createIfNotExists(builder.build());
    }

    /**
     * @deprecated CHEMODAN-82785 - Удалить createSubscriptionOrder как только обкатаем функционал.
     */
    @Deprecated
    public Order createSubscriptionOrder(PassportUid uid, String trustOrderId, UserProductPrice price) {
        logger.debug("createSubscriptionOrder uid=[{}] trustOrderId=[{}] price=[{}]", uid, trustOrderId, price);
        return orderDao.createOrUpdate(buildOrderInsertData(trustOrderId, uid, price));
    }

    private OrderDao.InsertData buildOrderInsertData(String trustOrderId, PassportUid uid, UserProductPrice price) {
        return OrderDao.InsertData.builder()
                .trustOrderId(trustOrderId)
                .type(OrderType.SUBSCRIPTION)
                .uid(uid.toString())
                .userProductPriceId(price.getId())
                .trustServiceId(price.getPeriod().getUserProduct().getTrustServiceId()
                        .orElseThrow(() -> new NoSuchElementException("trust service id not defined for: " + price)))
                .build();
    }

    public void processByTrustOrderId(String trustOrderId) {
        Option<Order> order = orderDao.findByTrustOrderId(trustOrderId);
        processOrder(order.orElseThrow(() -> new NotFoundException("Order " + trustOrderId + " not found")));
    }

    public void processByOrderIdAndState(UUID orderId, OrderStatus expectedStatus) {
        Order order = orderDao.findById(orderId);
        if (!Objects.equals(order.getStatus(),(expectedStatus))) {
            logger.info("Order {} current status is {} but expected status is {}",
                    order.getId(), order.getStatus(), expectedStatus);
            return;
        }
        processOrder(order);
    }

    public void processByOrderId(UUID orderId) {
        Order order = orderDao.findById(orderId);
        processOrder(order);
    }

    public ProcessInappReceiptResponse checkInappReceipt(PassportUid uid, InappStore inappStoreType,
                                                         String currency, String receipt) {
        return inappSubscriptionProcessor.checkInappReceipt(uid, inappStoreType, currency, receipt);
    }

    private OrderProcessor getProcessor(Order order) {
        switch (order.getType()) {
            case SUBSCRIPTION:
            case ORDER:
                return trustSubscriptionProcessor;
            case INAPP_SUBSCRIPTION:
                return inappSubscriptionProcessor;
            case PROMOCODE_ORDER:
                return promocodeOrderProcessor;
            default:
                throw new IllegalStateException("unable to define order processor for " + order);
        }
    }

    public AppleReceiptResponse checkAppstoreReceipt(String receipt) {
        return inappSubscriptionProcessor.checkAppstoreReceipt(receipt);
    }
}
