package ru.yandex.chemodan.app.psbilling.core.billing.users.processors;

import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Period;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.billing.users.BillingActionsReportingService;
import ru.yandex.chemodan.app.psbilling.core.config.Settings;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.users.OrderDao;
import ru.yandex.chemodan.app.psbilling.core.dao.users.UserServiceDao;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderType;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserServiceBillingStatus;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPrice;
import ru.yandex.chemodan.app.psbilling.core.promos.PromoService;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.users.UserService;
import ru.yandex.chemodan.app.psbilling.core.users.UserServiceManager;
import ru.yandex.chemodan.app.psbilling.core.util.LockService;
import ru.yandex.commune.bazinga.BazingaTaskManager;

/**
 * for billing type = free and order type = order
 */
@Slf4j
public class PromocodeOrderProcessor extends AbstractSubscriptionProcessor implements OrderProcessor {
    public PromocodeOrderProcessor(BazingaTaskManager bazingaTaskManager, OrderDao orderDao,
                                   UserServiceManager userServiceManager, LockService lockService,
                                   UserProductManager userProductManager, TaskScheduler taskScheduler,
                                   UserServiceDao userServiceDao, PromoService promoService, Settings settings,
                                   FeatureFlags featureFlags, BillingActionsReportingService billingActionsReportingService) {
        super(bazingaTaskManager, null, orderDao, userServiceManager, lockService, userProductManager, taskScheduler,
                userServiceDao, promoService, settings, featureFlags, billingActionsReportingService);
    }

    @Override
    public void processOrder(Order order) {
        if(!OrderType.PROMOCODE_ORDER.equals(order.getType())){
            throw new IllegalArgumentException("Can only process orders of type promocode. got order " + order);
        }

        Option<Instant> dueDateO =
                order.getUserServiceId().map(userServiceManager::findById).flatMapO(UserService::getDueDate);
        if (dueDateO.map(Instant::isBeforeNow).orElse(false)) {
            disableServiceIfNeeded(order);
        } else {
            enableService(order);
        }
    }

    private void enableService(Order order) {
        Option<UserService> userServiceO = order.getUserServiceId().map(userServiceManager::findById);
        Option<Target> existingTarget = userServiceO.map(UserService::getTarget);
        if (existingTarget.containsTs(Target.ENABLED)) {
            log.info("No need to enable service for order. {}. Service {} is already enabled", order,
                    userServiceO.orElse((UserService) null));
            return;
        }

        UserProductPrice price = userProductManager.findPrice(order.getUserProductPriceId());
        ListF<UserService> conflictingService = userServiceManager.findEnabledConflicting(
                order.getUid(),
                price.getPeriod().getUserProductId()
        );
        if (conflictingService.size() > 0) {
            throw new IllegalArgumentException("Upgrade case not supported for free promo products");
        }
        Period jop = price.getPeriod().getPeriod().toJodaPeriod();

        createUserServiceForOrder(order, DateTime.now().plus(jop).toInstant(), UserServiceBillingStatus.FREE_PERIOD, 0);
        billingActionsReportingService.builder(BillingActionsReportingService.Action.PROMOCODE_SPACE)
                .status("success")
                .order(order)
                .userProductPrice(price)
                .userProduct(price.getPeriod().getUserProduct())
                .finish();
    }

    private void disableServiceIfNeeded(Order order) {
        UserService us = userServiceManager.findById(order.getUserServiceId().orElseThrow());
        processStopSubscription(
                order,
                us.getDueDate().orElse(Instant.now().minus(Duration.standardSeconds(1))), // no due date -> disable immediately
                order.getSubscriptionsCount()
        );
    }
}
