package ru.yandex.chemodan.app.psbilling.core.billing.users.processors;

import java.util.function.BiConsumer;

import lombok.AllArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.chemodan.app.psbilling.core.config.Settings;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.util.LockService;
import ru.yandex.chemodan.trust.client.TrustException;
import ru.yandex.chemodan.trust.client.responses.InappSubscription;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
class TrustInappExceptionHandler {
    private static final Logger logger = LoggerFactory.getLogger(TrustInappExceptionHandler.class);
    private final Settings settings;
    private final LockService lockService;
    private final BiConsumer<Order, InappSubscription> forceDisableService;

    public void handleTrustException(TrustException trustException, InappSubscription subscription, Order order,
                                     boolean lockOrder) {
        if (!handleSubscriptionExpiredException(order, subscription, trustException, lockOrder)
                && !handleInvalidReceiptException(order, subscription, trustException, lockOrder)) {
            throw trustException;
        }
    }


    private boolean handleSubscriptionExpiredException(Order order, InappSubscription subscription,
                                                       TrustException trustException, boolean lockOrder) {
        if (trustException.getStatusCode().equals("play_subscription_not_found")) {
            logger.info("subscription {} has expired over 60 days ago. User service will be disabled",
                    subscription.getSubscriptionId());
            disableService(order, subscription, lockOrder);
            return true;
        }
        return false;
    }

    private boolean handleInvalidReceiptException(Order order, InappSubscription subscription,
                                                  TrustException trustException, boolean lockOrder) {
        if (trustException.getStatusCode().equals("invalid_receipt")) {
            if (subscription.getSubscriptionUntil().isBefore(Instant.now().minus(settings.getAcceptableUserServiceCheckDateExpirationTime()))) {
                logger.warn("trust subscription {} for order {} has subscribed until = {} " +
                                "(expired for more than {} hours) and trust says then receipt is invalid. " +
                                "Service will be disabled", subscription.getSubscriptionId(), order.getId(),
                        subscription.getSubscriptionUntil(),
                        settings.getAcceptableUserServiceCheckDateExpirationTime());
                disableService(order, subscription, lockOrder);
            } else {
                logger.warn("trust subscription {} for order {} has subscribed until = {} " +
                                "(expired for less than {} hours) and trust says then receipt is invalid. " +
                                "Service will not be disabled yet", subscription.getSubscriptionId(), order.getId(),
                        subscription.getSubscriptionUntil(),
                        settings.getAcceptableUserServiceCheckDateExpirationTime());
            }
            return true;
        } else {
            return false;
        }
    }

    private void disableService(Order order, InappSubscription subscription, boolean lockOrder) {
        if (lockOrder) {
            lockService.doWithOrderLockedInTransaction(order.getId(),
                    lockedOrder -> forceDisableService.accept(lockedOrder, subscription));
        } else {
            forceDisableService.accept(order, subscription);
        }
    }
}
