package ru.yandex.chemodan.app.psbilling.core.billing.users.processors;

import java.util.Comparator;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.UUID;

import org.apache.commons.lang3.mutable.MutableObject;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Period;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.psbilling.core.billing.users.BillingActionsReportingService;
import ru.yandex.chemodan.app.psbilling.core.billing.users.RefundOrderTask;
import ru.yandex.chemodan.app.psbilling.core.billing.users.TrustRefundService;
import ru.yandex.chemodan.app.psbilling.core.billing.users.UserUpgradeTask;
import ru.yandex.chemodan.app.psbilling.core.config.Settings;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.users.OrderDao;
import ru.yandex.chemodan.app.psbilling.core.dao.users.UserServiceDao;
import ru.yandex.chemodan.app.psbilling.core.entities.products.BillingType;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserServiceBillingStatus;
import ru.yandex.chemodan.app.psbilling.core.mail.EventMailType;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPrice;
import ru.yandex.chemodan.app.psbilling.core.promos.PromoService;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.users.UserService;
import ru.yandex.chemodan.app.psbilling.core.users.UserServiceManager;
import ru.yandex.chemodan.app.psbilling.core.util.LockService;
import ru.yandex.chemodan.trust.client.DurationUtils;
import ru.yandex.chemodan.trust.client.TrustClient;
import ru.yandex.chemodan.trust.client.requests.OrderRequest;
import ru.yandex.chemodan.trust.client.requests.PaymentRequest;
import ru.yandex.chemodan.trust.client.responses.InstantInterval;
import ru.yandex.chemodan.trust.client.responses.PaymentResponse;
import ru.yandex.chemodan.trust.client.responses.PaymentStatus;
import ru.yandex.chemodan.trust.client.responses.SubscriptionResponse;
import ru.yandex.chemodan.util.date.DateTimeUtils;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static ru.yandex.chemodan.app.psbilling.core.billing.users.processors.TrustSubscriptionProcessor.State.GRACE_PERIOD;
import static ru.yandex.chemodan.app.psbilling.core.billing.users.processors.TrustSubscriptionProcessor.State.INIT_NOT_PAID;
import static ru.yandex.chemodan.app.psbilling.core.billing.users.processors.TrustSubscriptionProcessor.State.INIT_PAY_ERROR;
import static ru.yandex.chemodan.app.psbilling.core.billing.users.processors.TrustSubscriptionProcessor.State.ON_HOLD;
import static ru.yandex.chemodan.app.psbilling.core.billing.users.processors.TrustSubscriptionProcessor.State.SUBSCRIPTION_AND_SERVICE_ACTIVE;
import static ru.yandex.chemodan.app.psbilling.core.billing.users.processors.TrustSubscriptionProcessor.State.SUBSCRIPTION_AND_SERVICE_IN_TRIAL;
import static ru.yandex.chemodan.app.psbilling.core.billing.users.processors.TrustSubscriptionProcessor.State.SUBSCRIPTION_STOPPED;


public class TrustSubscriptionProcessor extends AbstractSubscriptionProcessor implements OrderProcessor {
    private static final Logger logger = LoggerFactory.getLogger(TrustSubscriptionProcessor.class);
    private static final SetF<PaymentStatus> paymentInitStatuses =
            Cf.set(PaymentStatus.started, PaymentStatus.not_started, PaymentStatus.unknown);
    private static final SetF<PaymentStatus> paymentSuccessStatuses =
            Cf.set(PaymentStatus.authorized, PaymentStatus.cleared);
    private static final SetF<PaymentStatus> paymentErrorStatuses =
            Cf.set(PaymentStatus.cancelled, PaymentStatus.not_authorized, PaymentStatus.refunded);

    private final DynamicProperty<Integer> ordersWithoutPaymentsTimeoutMinutes =
            new DynamicProperty<>("orders.without.payments.timeout_in_minutes", 60);

    private final DynamicProperty<Integer> trustIntervalsThresholdMinutes =
            new DynamicProperty<>("ps-billing.trust.intervals.threshold_in_minutes", 5);

    TrustRefundService trustRefundService;

    public TrustSubscriptionProcessor(BazingaTaskManager bazingaTaskManager, TrustClient trustClient, OrderDao orderDao,
                                      UserServiceManager userServiceManager, LockService lockService,
                                      UserProductManager userProductManager,
                                      TaskScheduler taskScheduler, UserServiceDao userServiceDao,
                                      PromoService promoService, Settings settings, FeatureFlags featureFlags,
                                      BillingActionsReportingService billingActionsReportingService,
                                      TrustRefundService trustRefundService) {
        super(bazingaTaskManager, trustClient, orderDao, userServiceManager, lockService, userProductManager,
                taskScheduler, userServiceDao, promoService, settings, featureFlags, billingActionsReportingService);
        this.trustRefundService = trustRefundService;
    }

    @Override
    public void processOrder(Order order) {
        if (needToSkipOrder(order)) {
            return;
        }

        AnalyzedOrder analyzedOrder = analyzeSubscription(order);
        logger.info("analyzed order: {}", analyzedOrder);
        performAction(analyzedOrder);
    }

    private boolean needToSkipOrder(Order order) {
        if (order.getStatus() == OrderStatus.UPGRADED) {
            logger.info("Order {} was skipped because it was upgraded to {}", order.getId(),
                    order.getUpgradedOrderIdTo());
            return true;
        }

        return false;
    }

    private AnalyzedOrder analyzeSubscription(Order order) {
        logger.info("Analyzing order {}", order);
        PassportUid uid = PassportUid.cons(Long.parseLong(order.getUid()));
        SubscriptionResponse subscription = trustClient.getSubscription(OrderRequest.builder()
                .uid(uid).orderId(order.getTrustOrderId()).trustServiceId(order.getTrustServiceId()).build());
        logger.info("Trust subscription: {}", subscription);

        if (subscription.getSubscriptionPeriodCount() == 0) {
            Option<PaymentResponse> paymentO = subscription.getPaymentIds().lastO().map(
                    paymentId -> trustClient.getPayment(PaymentRequest.builder()
                            .trustServiceId(order.getTrustServiceId()).uid(uid).purchaseToken(paymentId)
                            .build())
            );
            logger.info("last payment for subscription: {}", paymentO);

            if (!paymentO.isPresent()) {
                if (order.getCreatedAt().plus(Duration.standardMinutes(ordersWithoutPaymentsTimeoutMinutes.get()))
                        .isBeforeNow()) {
                    return consAnalyzedOrder(INIT_PAY_ERROR, order, subscription, "payment_not_created", null);
                } else {
                    return consAnalyzedOrder(INIT_NOT_PAID, order, subscription);
                }
            }

            PaymentResponse payment = paymentO.get();

            billingActionsReportingService.builder(BillingActionsReportingService.Action.PROCESS_ORDER)
                    .order(order)
                    .subscriptionResponse(subscription)
                    .paymentResponse(payment)
                    .finish();

            if (paymentInitStatuses.containsTs(payment.getPaymentStatus())) {
                // CHEMODAN-78555: Обработка зависших платежей траста
                if (order.getCreatedAt().plus(settings.getAcceptableInitOrdersCheckDateExpirationTime()).isBeforeNow()) {
                    logger.warn("order {} payment is in initial state for too long (more than {} hours). " +
                                    "Order will be marked as error", order.getId(),
                            settings.getAcceptableInitOrdersCheckDateExpirationTime());
                    return consAnalyzedOrder(INIT_PAY_ERROR, order, subscription,
                            payment.getPaymentStatus() + "_hanged",
                            "timeout while waiting for terminal payment status");
                }
                return consAnalyzedOrder(INIT_NOT_PAID, order, subscription);
            } else if (paymentSuccessStatuses.containsTs(payment.getPaymentStatus())) {
                logger.info("subscription not active, but payment successful");
                return consAnalyzedOrder(INIT_NOT_PAID, order, subscription);
            } else if (paymentErrorStatuses.containsTs(payment.getPaymentStatus())) {
                return consAnalyzedOrder(INIT_PAY_ERROR, order, subscription, payment.getPaymentStatusCode(),
                        payment.getPaymentStatusDescription());
            } else {
                throw new IllegalStateException("unreachable");
            }
        }

        if (subscription.getSubscriptionPeriodCount() > 0) {
            if (subscription.getFinishTime() != null) {
                return consAnalyzedOrder(SUBSCRIPTION_STOPPED, order, subscription);
            } else {
                if (isInGrace(subscription)) {
                    return consAnalyzedOrder(GRACE_PERIOD, order, subscription);
                }
                if (isOnHold(subscription)) {
                    return consAnalyzedOrder(ON_HOLD, order, subscription);
                }
                if (subscription.getSubscriptionState().isInTrial()) {
                    return consAnalyzedOrder(SUBSCRIPTION_AND_SERVICE_IN_TRIAL, order, subscription);
                }
                if (isStale(order, subscription)) {
                    return consAnalyzedOrder(SUBSCRIPTION_STOPPED, order, subscription);
                }
                return consAnalyzedOrder(SUBSCRIPTION_AND_SERVICE_ACTIVE, order, subscription);
            }
        }
        throw new IllegalStateException("unreachable, subscription: " + subscription);
    }


    private AnalyzedOrder consAnalyzedOrder(State state, Order order, SubscriptionResponse trustSubscription) {
        return consAnalyzedOrder(state, order, trustSubscription, Option.empty(), Option.empty());
    }

    private AnalyzedOrder consAnalyzedOrder(State state, Order order, SubscriptionResponse trustSubscription,
                                            String errorCode, String errorMessage) {
        return consAnalyzedOrder(state, order, trustSubscription, Option.ofNullable(errorCode),
                Option.ofNullable(errorMessage));
    }

    private AnalyzedOrder consAnalyzedOrder(State state, Order order, SubscriptionResponse trustSubscription,
                                            Option<String> errorCode, Option<String> errorMessage) {
        UserProductPrice price = userProductManager.findPrice(order.getUserProductPriceId());

        return new AnalyzedOrder(state, order, trustSubscription, price, errorCode, errorMessage);
    }

    private boolean isOnHold(SubscriptionResponse subscription) {
        if (subscription.getSubscriptionUntil().isAfter(Instant.now()))
            return false;
        ListF<InstantInterval> holdIntervals = subscription.getHoldIntervals();
        if (holdIntervals == null) {
            return false;
        }
        //looks like interval.to is trust current time. So it's possible to get a race and status flap if request is slowed for some reason
        return holdIntervals.exists(interval -> trustIntervalInRange(Instant.now(), interval));
    }

    private boolean isStale(Order order, SubscriptionResponse subscription) {
        if (!featureFlags.getDisableStaleWebServices().isEnabledForUid(PassportUid.cons(Long.parseLong(order.getUid()))))
            return false;
        //fix for trust bug with empty finishTime CHEMODAN-84596
        if (subscription.getFinishTime() != null)
            return false;
        if (subscription.getSubscriptionUntil().isAfter(Instant.now()))
            return false;
        if (subscription.getHoldIntervals() == null || subscription.getHoldIntervals().isEmpty())
            return false;
        if (isInGrace(subscription))
            return false;
        InstantInterval lastHoldInterval = subscription.getHoldIntervals()
                .max(Comparator.comparing(InstantInterval::getTo));
        Instant holdTo = lastHoldInterval.getTo();
        if (subscription.getSubscriptionUntil().isAfter(holdTo))
            return false;
        Duration threshold = settings.getAcceptableUserServiceCheckDateExpirationTime();
        if (Instant.now().minus(threshold).isAfter(holdTo))
            logger.warn("subscription for order {} is not in grace or hold and has subscribed until = {} " +
                            "(hold is over for more than {} hours) service should be disabled",
                    order.getId(),
                    subscription.getSubscriptionUntil(),
                    threshold);
        return true;
    }

    private boolean isInGrace(SubscriptionResponse subscription) {
        if (subscription.getGraceIntervals() == null) {
            return false;
        }

        //looks like interval.to is trust current time. So it's possible to get a race and status flap if request is slowed for some reason
        return subscription.getGraceIntervals().exists(interval -> trustIntervalInRange(Instant.now(), interval));
    }

    private void performAction(AnalyzedOrder analyzedOrder) {
        switch (analyzedOrder.state) {
            case INIT_NOT_PAID:
                onInitNotPaid(analyzedOrder);
                break;
            case INIT_PAY_ERROR:
                onInitPayError(analyzedOrder);
                break;
            case SUBSCRIPTION_AND_SERVICE_IN_TRIAL:
                onSubscriptionInTrial(analyzedOrder);
                break;
            case SUBSCRIPTION_AND_SERVICE_ACTIVE:
                onActiveServiceAndSubscription(analyzedOrder);
                break;
            case GRACE_PERIOD:
                onGracePeriod(analyzedOrder);
                break;
            case ON_HOLD:
                onHold(analyzedOrder);
                break;
            case SUBSCRIPTION_STOPPED:
                onSubscriptionStopped(analyzedOrder);
                break;
            default:
                throw new NoSuchElementException("Unexpected state: " + analyzedOrder.state);
        }
    }

    enum State {
        INIT_NOT_PAID, // подписка еще не подключена, ждем платежа
        INIT_PAY_ERROR, // первичная оплата подписки не прошла (ошибка оплаты)
        SUBSCRIPTION_AND_SERVICE_IN_TRIAL, // подписка в триальном периоде
        SUBSCRIPTION_AND_SERVICE_ACTIVE, // подписка активна и подключена
        GRACE_PERIOD, // подписка активна, пытаемся списать деньги (идет грейс-период)
        ON_HOLD, // подписка отключена, пытаемся списать деньги и восстановить подписку
        SUBSCRIPTION_STOPPED // подписка отменена, но услуга оказывается до какой-то даты
    }

    static class AnalyzedOrder extends DefaultObject {
        private final State state;
        private final Order orderUnlocked;
        private final SubscriptionResponse trustSubscription;
        private final UserProductPrice price;
        private final UserProduct product;
        private final Option<String> paymentErrorCode;
        private final Option<String> getPaymentErrorMessage;

        public AnalyzedOrder(State state, Order order, SubscriptionResponse trustSubscription,
                             UserProductPrice price, Option<String> errorCode, Option<String> errorMessage) {
            this.state = state;
            this.trustSubscription = trustSubscription;
            this.orderUnlocked = order;
            this.price = price;
            this.paymentErrorCode = errorCode;
            this.getPaymentErrorMessage = errorMessage;
            this.product = price.getPeriod().getUserProduct();
        }
    }

    private void onSubscriptionInTrial(AnalyzedOrder analyzedOrder) {
        logger.info("onSubscriptionInTrial: {}", analyzedOrder);
        createOrProlongService(analyzedOrder, UserServiceBillingStatus.FREE_PERIOD);
    }

    private void onActiveServiceAndSubscription(AnalyzedOrder analyzedOrder) {
        logger.info("onActiveServiceAndSubscription: {}", analyzedOrder);
        createOrProlongService(analyzedOrder, UserServiceBillingStatus.PAID);
    }

    @SuppressWarnings({"EmptyMethod", "unused"})
    private void onInitNotPaid(AnalyzedOrder order) {
        // do nothing
    }

    private void onGracePeriod(AnalyzedOrder analyzedOrder) {
        logger.info("onGracePeriod: {}", analyzedOrder);
        MutableObject<UUID> userServiceId = new MutableObject<>();
        Instant now = Instant.now();
        String uid = analyzedOrder.orderUnlocked.getUid();
        lockService.doWithUserLockedInTransaction(uid, () -> {
            Order order = orderDao.findById(analyzedOrder.orderUnlocked.getId());
            UUID serviceId = order.getUserServiceId().orElseThrow(() -> {
                logger.error("order {} without service, but in grace period", order);
                return new IllegalStateException("order without service, but in grace period");
            });
            userServiceId.setValue(serviceId);

            UserProduct product = userServiceManager.findById(serviceId).getUserProduct();

            Tuple2<Instant, Instant> dueAndNextCheckDates = calcDueDateAndNextCheckDateForGrace(
                    analyzedOrder.trustSubscription.getGraceIntervals(), product, now);

            userServiceManager.updateSubscriptionStatus(
                    serviceId, UserServiceBillingStatus.WAIT_AUTO_PROLONG,
                    dueAndNextCheckDates.get1(), dueAndNextCheckDates.get2());

            //сигнализирует что это первая проверка в рамках грейс-периода
            if (isFirstGraceRetry(analyzedOrder, now)) {
                PassportUid passportUid = PassportUid.cons(Long.parseLong(uid));
                if (featureFlags.getLongBillingPeriodForTrustEmailEnabled().isEnabledForUid(passportUid)) {
                    taskScheduler.scheduleInGraceEmailTask(order);
                } else {
                    taskScheduler.scheduleTransactionalEmailTask(
                            EventMailType.SUBSCRIPTION_HAS_NOT_BEEN_PAID,
                            buildGraceEmailContext(userServiceId.getValue(), uid));
                }
            }
        });
    }

    private boolean isFirstGraceRetry(AnalyzedOrder analyzedOrder, Instant now) {
        Option<Period> retryDelayO =
                analyzedOrder.product.getTrustSubsChargingRetryDelay().map(DurationUtils::parsePeriod);
        if (!retryDelayO.isPresent()) {
            return false;
        }
        Period retryDelay = retryDelayO.get();
        InstantInterval currentInterval = analyzedOrder.trustSubscription.getGraceIntervals()
                .filter(graceInterval -> trustIntervalInRange(now, graceInterval))
                .first();
        return now.isBefore(currentInterval.getFrom().plus(retryDelay.toDurationFrom(now)));
    }

    private void onHold(AnalyzedOrder analyzedOrder) {
        Order orderUnlocked = analyzedOrder.orderUnlocked;
        SubscriptionResponse subscription = analyzedOrder.trustSubscription;
        logger.error("HOLD order {}, subscription: {}", orderUnlocked, subscription);
        processHoldOrder(orderUnlocked, subscription.getSubscriptionUntil(), subscription.getSubscriptionPeriodCount());
    }

    private void onSubscriptionStopped(AnalyzedOrder analyzedOrder) {
        logger.info("onSubscriptionStopped: {}", analyzedOrder);
        processStopSubscription(analyzedOrder.orderUnlocked,
                analyzedOrder.trustSubscription.getSubscriptionUntil(),
                analyzedOrder.trustSubscription.getSubscriptionPeriodCount());
    }

    private void onInitPayError(AnalyzedOrder order) {
        logger.info("onInitPayError: {}", order);
        String code = order.paymentErrorCode.orElse((String) null);
        String message = order.getPaymentErrorMessage.orElse((String) null);
        orderDao.writeErrorStatus(order.orderUnlocked.getId(), code, message);

        Option<UUID> productId = Option.of(order.product.getId());
        PassportUid uid = PassportUid.cons(Long.parseLong(order.orderUnlocked.getUid()));

        taskScheduler.scheduleAbandonedCartEmailTask(uid, productId);
        if (!featureFlags.getPleaseComeBackPromoOnTuning().isEnabled()) {
            taskScheduler.schedulePleaseComeBackTask(uid, productId.get());
        }

        billingActionsReportingService.builder(BillingActionsReportingService.Action.BUY_NEW)
                .status("payment_failed")
                .order(order.orderUnlocked)
                .userProductPrice(order.price)
                .userProduct(order.product)
                .errorCode(code)
                .errorMessage(message)
                .finish();
    }

    private void createOrProlongService(AnalyzedOrder analyzedOrder, UserServiceBillingStatus serviceBillingStatus) {
        int subscriptionsCount = analyzedOrder.trustSubscription.getSubscriptionPeriodCount();

        createOrProlongServiceForOrder(analyzedOrder.orderUnlocked,
                analyzedOrder.price,
                analyzedOrder.trustSubscription.getSubscriptionUntil(),
                subscriptionsCount,
                serviceBillingStatus);
    }

    private Tuple2<Instant, Instant> calcDueDateAndNextCheckDateForGrace(
            ListF<InstantInterval> intervals, UserProduct product, Instant now) {
        Period delay = DurationUtils.parsePeriod(product.getTrustSubsChargingRetryDelay().orElseThrow(
                () -> new IllegalStateException("subscription in grace period, but product hasn't retry delay")));

        Period period = DurationUtils.parsePeriod(product.getTrustSubsGracePeriod().orElseThrow(
                () -> new IllegalStateException("subscription in grace period, but product has no period specified")));

        InstantInterval interval = intervals.filter(i -> trustIntervalInRange(now, i)).first();

        Instant nextCheckDate = interval.getFrom();
        while (!nextCheckDate.isAfter(now)) {
            nextCheckDate = nextCheckDate.plus(delay.toDurationFrom(nextCheckDate));
        }
        return new Tuple2<>(
                interval.getFrom().plus(period.toDurationFrom(interval.getFrom())),
                nextCheckDate.isAfter(interval.getTo()) ? interval.getTo() : nextCheckDate
        );
    }

    private boolean trustIntervalInRange(Instant now, InstantInterval interval) {
        return DateTimeUtils.isInRange(
                Option.ofNullable(interval.getFrom()),
                Option.ofNullable(interval.getTo())
                        .map(to -> to.plus(Duration.standardMinutes(trustIntervalsThresholdMinutes.get()))),
                now);
    }

    private Order prepareTrustUpgrade(Order orderUpgradeTo, UserService oldService) {
        Order conflictingOrder = orderDao.findById(oldService.getLastPaymentOrderId().get());

        userServiceManager.disableService(oldService.getId());
        userServiceDao.stopAutoProlong(oldService.getId());
        return orderDao.upgradeOrder(conflictingOrder.getId(), orderUpgradeTo.getId());
    }

    private void createOrProlongServiceForOrder(
            Order orderUnlocked, UserProductPrice price, Instant subscribedUntil, int subscriptionsCount,
            UserServiceBillingStatus serviceBillingStatus) {
        lockService.doWithUserLockedInTransaction(orderUnlocked.getUid(), () -> {
            Order order = orderDao.findById(orderUnlocked.getId());
            logger.info("order from db: {}", order);
            UserProduct product = price.getPeriod().getUserProduct();
            if (!Objects.equals(order.getUserProductPriceId(), price.getId())) {
                throw new IllegalStateException("price of the order changed");
            }

            Option<UserService> serviceO = order.getUserServiceId().map(userServiceManager::findById);
            boolean activeServiceExists = serviceO.map(UserService::getTarget).containsTs(Target.ENABLED);
            if (activeServiceExists) {
                UserService userService = serviceO.get();
                prolongServiceWithoutLocking(order, price, userService, subscribedUntil, subscriptionsCount,
                        serviceBillingStatus);
                return;
            }
            if (trustRefundService.isRefunded(order)) {
                logger.warn("Order refunded. Skip processing");
                return;
            }

            ListF<UserService> conflictingService = userServiceManager.findEnabledConflicting(order.getUid(), product.getId());

            if (conflictingService.isEmpty()) {
                UserServiceCreatedResult serviceReport = createUserServiceForOrder(order, subscribedUntil,
                        serviceBillingStatus, subscriptionsCount);
                logNewServiceCreated(price, order, product, Option.empty(),
                        serviceReport.userService.getNextCheckDate().get(), serviceReport.usedPromoTemplateCode);
                return;
            }
            Option<UserService> serviceForUpgrade = userServiceManager.findServiceForUpgrade(
                    order.getUid(), BillingType.TRUST, Option.empty(), Cf.set(product.getId()));

            if (serviceForUpgrade.isEmpty()
                    || conflictingService.stream().noneMatch(x -> x.getId().equals(serviceForUpgrade.get().getId()))) {
                // got conflicts but upgrade not available - current order have to be refunded CHEMODAN-78219
                refundOrder(order, subscriptionsCount);
                return;
            }

            UserService oldService = serviceForUpgrade.get();
            UserProductPrice oldPrice = oldService.getPrice().orElseThrow(
                    () -> new IllegalStateException("conflicting service hasn't price, cannot refund or " +
                            "upgrade"));
            // https://st.yandex-team.ru/CHEMODAN-76805#605090ec7a92055f02f0b19a
            // если идет покупка более дешевой или такой же подписки, то старую новую покупку рефандим
            // если идет покупка более дорогой подписки, то делаем апгрейд
            if (oldPrice.getPrice().compareTo(price.getPrice()) >= 0) {
                refundOrder(order, subscriptionsCount);
                return;
            }
            prepareTrustUpgrade(order, oldService);
            UserServiceCreatedResult serviceReport = createUserServiceForOrder(order, subscribedUntil,
                    serviceBillingStatus, subscriptionsCount, false);
            Instant nextCheckDate = userServiceManager.addSupplementPeriod(serviceReport.userService, order,
                    oldService, false);
            logNewServiceCreated(price, order, product, serviceForUpgrade, nextCheckDate,
                    serviceReport.usedPromoTemplateCode
            );
            bazingaTaskManager.schedule(new UserUpgradeTask(serviceReport.userService.getId(),
                    order.getId(), oldService.getId(), oldService.getLastPaymentOrderId().get()));
        });
    }

    private void refundOrder(Order order, int subscriptionsCount) {
        logger.info("Order {} will be mark as paid and refunded", order);
        orderDao.onSuccessfulOrderPurchase(order.getId(), Option.empty(), subscriptionsCount);
        bazingaTaskManager.schedule(new RefundOrderTask(order.getId()));
    }

    private void logNewServiceCreated(UserProductPrice price, Order order, UserProduct product,
                                      Option<UserService> oldServiceO, Instant nextCheckDate,
                                      Option<String> activePromoTemplateCode) {

        BillingActionsReportingService.Action action;
        if (oldServiceO.isPresent()) {
            action = BillingActionsReportingService.Action.ORDER_UPGRADED;
        } else if (order.getStatus() == OrderStatus.ON_HOLD)
            action = BillingActionsReportingService.Action.RESTORE_FROM_HOLD;
        else {
            action = BillingActionsReportingService.Action.BUY_NEW;
        }
        Option<Order> oldOrder = oldServiceO.flatMapO(UserService::getLastPaymentOrderId).map(orderDao::findById);
        CalculatedPrice priceValue = getPriceForPeriod(price, order);
        billingActionsReportingService.builder(action)
                .status("success")
                .order(order)
                .userProductPrice(price)
                .userProduct(product)
                .oldOrderId(oldOrder.map(Order::getTrustOrderId).orElse((String) null))
                .oldServiceId(oldServiceO.map(UserService::getId).map(UUID::toString).orElse((String) null))
                .newExpirationDate(nextCheckDate)
                .calculatedPrice(priceValue)
                .active_promo(activePromoTemplateCode.orElse((String) null))
                .finish();
    }

    @Override
    protected CalculatedPrice getPriceForPeriod(UserProductPrice price, Order order) {
        if (price.getStartPeriodPrice().isEmpty() && !price.getPeriod().hasStartPeriod())
            return new CalculatedPrice(price.getPrice());
        if (price.getPeriod().getStartPeriodCount().get() > order.getSubscriptionsCount())
            return new CalculatedPrice(price.getStartPeriodPrice().get(), true);
        return new CalculatedPrice(price.getPrice());

    }
}
