package ru.yandex.chemodan.app.psbilling.core.config;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.List;
import java.util.Optional;

import org.apache.commons.httpclient.ProxyHost;
import org.apache.http.impl.client.CloseableHttpClient;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.web.client.RestTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.GroupBalanceService;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.balanceclient.BalanceClient;
import ru.yandex.chemodan.balanceclient.BalanceXmlRpcClient;
import ru.yandex.chemodan.balanceclient.BalanceXmlRpcClientConfig;
import ru.yandex.chemodan.balanceclient.model.request.BalanceClientRequstFactory;
import ru.yandex.chemodan.boot.ChemodanInitContextConfiguration;
import ru.yandex.chemodan.boot.value.OverridableValuePrefix;
import ru.yandex.chemodan.http.HttpClientWithThreadLocalVariable;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.chemodan.zk.registries.tvm.TvmDstClientsZkRegistry;
import ru.yandex.chemodan.zk.registries.tvm.ZkTvm2ContextConfiguration;
import ru.yandex.inside.passport.tvm2.Tvm2;

@Configuration
@Import({
        ZkTvm2ContextConfiguration.class,
        ChemodanInitContextConfiguration.class,
        PsBillingTrustConfiguration.class,
        CommonServicesConfig.class
})
public class PsBillingBalanceConfiguration {
    @Bean
    public BalanceXmlRpcClientConfig balanceXmlRpcClientConfig(
            Tvm2 tvm2, TvmDstClientsZkRegistry registry,
            //bean defined only in tests
            @Qualifier("balanceClientProxy") Optional<ProxyHost> balanceClientProxy) {
        return new BalanceXmlRpcClientConfig(
                tvm2, host -> registry.getO(host).flatMapO(i -> i.clientId), Option.x(balanceClientProxy));
    }

    @Bean
    public BalanceClientRequstFactory balanceClientRequstFactory() {
        return new BalanceClientRequstFactory();
    }

    @Bean
    public BalanceClient balanceClient(BalanceXmlRpcClientConfig balanceXmlRpcClientConfig,
                                       BalanceClientRequstFactory requestFactory) {
        return new BalanceClient(new BalanceXmlRpcClient(balanceXmlRpcClientConfig), requestFactory);
    }

    @Bean(name = {"balanceRestTemplate"})
    public RestTemplate balanceRestTemplate(@Balance HttpClientConfigurator balanceHttpClientConfigurator) {
        CloseableHttpClient httpClient = balanceHttpClientConfigurator.createBuilder()
                .multiThreaded()
                .build();
        CloseableHttpClient wrapped = new HttpClientWithThreadLocalVariable(httpClient, "name", "balance_rest");

        return new RestTemplate(new HttpComponentsClientHttpRequestFactory(wrapped));
    }

    @Bean
    public BalanceService balanceService(BalanceClient balanceClient, @Value("${balance.service-id}") Integer serviceId,
                                         BalanceOfferConfig offerConfig,
                                         @Value("${balance.url}") String balanceUrl,
                                         @Value("${group.balance.notification.url_template}") String groupPaymentNotificationUrlTemplate,
                                         @Value("${group.balance.binding_notification.url_template}") String groupCardBindingNotificationUrlTemplate,
                                         @Value("${tvm.self.client-id}") String selfClientId,
                                         List<PsBillingTrustConfiguration.TrustServiceDefinition> services,
                                         @Qualifier("balanceRestTemplate") RestTemplate balanceRestTemplate) {
        return new BalanceService(balanceClient, serviceId, offerConfig, balanceRestTemplate, balanceUrl,
                groupPaymentNotificationUrlTemplate, groupCardBindingNotificationUrlTemplate, selfClientId,
                Cf.x(services).toMap(PsBillingTrustConfiguration.TrustServiceDefinition::getServiceId,
                        PsBillingTrustConfiguration.TrustServiceDefinition::getServiceToken));
    }

    @Bean
    public GroupBalanceService groupBalanceService(BalanceService balanceService, FeatureFlags featureFlags) {
        return new GroupBalanceService(balanceService, featureFlags);
    }

    @Bean
    @OverridableValuePrefix("balance.create-offer-params")
    public BalanceOfferConfig offerConfig() {
        return new BalanceOfferConfig();
    }


    @Qualifier
    @Retention(RetentionPolicy.RUNTIME)
    public @interface Balance {
    }

    @Bean
    @Balance
    @OverridableValuePrefix("balance")
    public HttpClientConfigurator balanceHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }
}
