package ru.yandex.chemodan.app.psbilling.core.config;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.lang.Validate;


public class Settings {

    private final DynamicProperty<Integer> waitBeforeCancellingGroupPaymentExpirationInMinutes =
            new DynamicProperty<>("ps-billing.wait-before-cancelling-group-payment-after-expiration.minutes", 60);

    private final DynamicProperty<Integer> waitBeforeCancellingAfterSubscriptionExpirationInHours =
            new DynamicProperty<>("ps-billing.wait-before-cancelling-subscription-after-expiration.hours", 168);

    private final DynamicProperty<Integer> waitBeforeCancellingPaymentExpirationInHours =
            new DynamicProperty<>("ps-billing.wait-before-cancelling-payment-after-expiration.hours", 24);

    private final DynamicProperty<Integer> acceptableNotSyncedTimeInHours =
            new DynamicProperty<>("ps-billing.wait-before-resync-inapp-subscription.hours", 24);

    private final DynamicProperty<ListF<String>> disabledPriceRegions =
            new DynamicProperty<>("ps-billing.disabled-price-regions", Cf.list());

    private final DynamicProperty<Integer> balanceStaleIntervalHours =
            new DynamicProperty<>("ps-billing.balance_stale_interval_hours", 24);

    private final DynamicProperty<Integer> balanceInvoiceStaleIntervalHours =
            new DynamicProperty<>("ps-billing.balance_invoice_stale_interval_hours", 24 * 7);

    private final DynamicProperty<Integer> staleCardBindingIntervalHours =
            new DynamicProperty<>("ps-billing.stale_card_binding_interval_hours", 2);

    // Auto payment
    // Payments start N hours before balance void
    private final DynamicProperty<Integer> autoPayLeadTimeMinutes =
            new DynamicProperty<>("ps-billing.auto_pay.lead_time_minutes", 6 * 60);
    // Skip payment if got fresh init payment
    private final DynamicProperty<Integer> autoPayRecentInitPaymentsLookupMinutes =
            new DynamicProperty<>("ps-billing.auto_pay.payments.last_init_lookup_minutes", 3 * 60);
    // Skip payment if gor fresh success payment
    private final DynamicProperty<Integer> autoPayRecentSuccessPaymentsLookupMinutes =
            new DynamicProperty<>("ps-billing.auto_pay.payments.last_success_lookup_minutes", 3 * 60);
    // use previous canceled payment to respect charge plan
    private final DynamicProperty<Integer> autoPayRecentCanceledPaymentsLookupMinutes =
            new DynamicProperty<>("ps-billing.auto_pay.payments.last_canceled_lookup_minutes", 3 * 60);
    // Month prices per all client services * coefficient. Retry plan on no enough funds
    private final DynamicProperty<ListF<Double>> autoPayChargePeriodPlan =
            new DynamicProperty<>("ps-billing.auto_pay.charge_period_coeff_plan", Cf.list(1.0, 0.5, 0.25),
                    this::validateChargePlan);
    // min interval to retry charge
    private final DynamicProperty<Integer> autoPayMinRetryIntervalMinutes =
            new DynamicProperty<>("ps-billing.auto_pay.min_retry_interval_minutes", 0);

    // Auto resurrect payment
    // Resurrect services no older then threshold
    private final DynamicProperty<Integer> autoResurrectMaxResurrectTimeMinutes =
            new DynamicProperty<>("ps-billing.auto_resurrect_pay.max_resurrect_time_minutes", 7 * 24 * 60);
    // min interval to retry charge
    private final DynamicProperty<Integer> autoResurrectMinRetryIntervalMinutes =
            new DynamicProperty<>("ps-billing.auto_resurrect_pay.min_retry_interval_minutes", 23 * 60);

    public Duration getAcceptableGroupPaymentExpirationTime() {
        return Duration.standardMinutes(waitBeforeCancellingGroupPaymentExpirationInMinutes.get());
    }

    public Duration getAcceptableUserServiceCheckDateExpirationTime() {
        return Duration.standardHours(waitBeforeCancellingAfterSubscriptionExpirationInHours.get());
    }

    public Duration getAcceptableInitOrdersCheckDateExpirationTime() {
        return Duration.standardHours(waitBeforeCancellingPaymentExpirationInHours.get());
    }

    public Duration getAcceptableNotSyncedTimeInHours() {
        return Duration.standardHours(acceptableNotSyncedTimeInHours.get());
    }

    public Duration getAutoPayLeadTime() {
        return Duration.standardMinutes(autoPayLeadTimeMinutes.get());
    }

    public Duration getAutoPayRecentInitPaymentsLookupTime() {
        return Duration.standardMinutes(autoPayRecentInitPaymentsLookupMinutes.get());
    }

    public Duration getAutoPayRecentSuccessPaymentsLookupTime() {
        return Duration.standardMinutes(autoPayRecentSuccessPaymentsLookupMinutes.get());
    }

    public Duration getAutoPayRecentCanceledPaymentsLookupTime() {
        return Duration.standardMinutes(autoPayRecentCanceledPaymentsLookupMinutes.get());
    }

    public SetF<String> getDisabledPriceRegions() {
        return Cf.toSet(disabledPriceRegions.get().map(String::trim));
    }

    public Duration getBalanceStaleIntervalHours() {
        return Duration.standardHours(balanceStaleIntervalHours.get());
    }

    public Duration getBalanceInvoiceStaleIntervalHours() {
        return Duration.standardHours(balanceInvoiceStaleIntervalHours.get());
    }

    public Duration getStaleCardBindingIntervalHours() {
        return Duration.standardHours(staleCardBindingIntervalHours.get());
    }

    public ListF<Double> getAutoPayChargePlan() {
        return autoPayChargePeriodPlan.get();
    }

    public Duration getAutoPayMinRetryIntervalMinutes() {
        return Duration.standardMinutes(autoPayMinRetryIntervalMinutes.get());
    }

    public Duration getAutoResurrectMaxResurrectTimeMinutes() {
        return Duration.standardMinutes(autoResurrectMaxResurrectTimeMinutes.get());
    }

    public Duration getAutoResurrectMinRetryIntervalMinutes() {
        return Duration.standardMinutes(autoResurrectMinRetryIntervalMinutes.get());
    }

    private boolean validateChargePlan(ListF<Double> value) {
        try {
            Validate.isTrue(value.size() == value.unique().size(), "charge plan periods have to be unique");
            Validate.isTrue(value.size() > 0, "no charge plan");
            return true;
        } catch (Exception ex) {
            return false;
        }
    }
}
