package ru.yandex.chemodan.app.psbilling.core.dao.cards.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.intellij.lang.annotations.Language;
import org.joda.time.Instant;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.cards.CardDao;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardPurpose;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardStatus;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class CardDaoImpl extends AbstractDaoImpl<CardEntity> implements CardDao {

    public CardDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "cards";
    }

    @Override
    public CardEntity parseRow(ResultSet rs) throws SQLException {
        return new CardEntity(
                UUID.fromString(rs.getString("id")),
                PassportUid.cons(Long.parseLong(rs.getString("uid"))),
                CardPurpose.R.fromValue(rs.getString("purpose")),
                rs.getString("external_id"),
                new Instant(rs.getTimestamp("created_at")),
                CardStatus.R.fromValue(rs.getString("status")),
                new Instant(rs.getTimestamp("updated_at"))
        );
    }

    @Override
    public Option<CardEntity> findByExternalId(PassportUid uid, String externalId) {
        return jdbcTemplate.queryForOption("select * from cards where uid = ? and external_id = ?",
                (rs, rowNum) -> parseRow(rs), uid.toString(), externalId);
    }

    @Override
    public ListF<CardEntity> findByPurpose(PassportUid uid, CardPurpose purpose) {
        return jdbcTemplate.query("select * from cards where uid = ? and purpose = ?::card_purpose",
                (rs, rowNum) -> parseRow(rs), uid.toString(), purpose.value());
    }

    @Override
    public Option<CardEntity> findB2BPrimary(PassportUid uid) {
        return findByPurpose(uid, CardPurpose.B2B_PRIMARY).filter(x -> x.getStatus().equals(CardStatus.ACTIVE)).firstO();
    }

    @Override
    public ListF<CardEntity> findCardsByUid(PassportUid uid) {
        return jdbcTemplate.query("select * from cards where uid = ?",
                (rs, rowNum) -> parseRow(rs), uid.toString());
    }

    @Override
    public CardEntity insert(InsertData insertData) {
        MapF<String, Object> params = makeParamsFromInsertData(insertData);

        return jdbcTemplate.query("insert into cards (uid, purpose, external_id, " +
                "created_at, status, updated_at ) values " +
                " (:uid, :purpose::card_purpose, :external_id, :now, :status::card_status, :now )" +
                " returning *", (rs, num) -> parseRow(rs), params).first();
    }

    @Override
    public CardEntity insertOrUpdateStatus(InsertData insertData) {
        MapF<String, Object> params = makeParamsFromInsertData(insertData);

        return jdbcTemplate.query("insert into cards (uid, purpose, external_id, " +
                "created_at, status, updated_at ) values " +
                " (:uid, :purpose::card_purpose, :external_id, :now, :status::card_status, :now )" +
                " on conflict (uid, external_id) " +
                " do update set status = :status::card_status, updated_at = :now" +
                " returning *", (rs, num) -> parseRow(rs), params).first();
    }

    @Override
    public CardEntity insertOrUpdateStatusWithCheckingOtherPrimary(InsertData insertData) {
        MapF<String, Object> params = makeParamsFromInsertData(insertData);
        params.put("new_purpose_if_primary_exists", CardPurpose.B2B.value());

        return jdbcTemplate.query("insert into cards (uid, purpose, external_id, " +
                "created_at, status, updated_at ) values " +
                "(:uid, :purpose::card_purpose, :external_id, :now, :status::card_status, :now) " +
                "on conflict (uid, external_id) " +
                "do update set purpose = (select " +
                "  case when exists (select * from cards " +
                "                    where uid = :uid " +
                "                      and purpose = 'b2b_primary'::card_purpose " +
                "                      and status = 'active'::card_status " +
                "                      and external_id != :external_id " +
                "                    limit 1) " +
                "       then (select case when cards.purpose = 'b2b_primary'::card_purpose " +
                "                    then :new_purpose_if_primary_exists::card_purpose " +
                "                    else cards.purpose end) " +
                "       else cards.purpose " +
                "  end), " +
                "status = :status::card_status, updated_at = :now " +
                "returning *", (rs, num) -> parseRow(rs), params).first();
    }

    @Override
    public CardEntity setB2BPrimaryCard(PassportUid uid, String externalId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("uid", uid.toString());
        params.put("external_id", externalId);
        params.put("purpose_b2b_primary", CardPurpose.B2B_PRIMARY.value());
        params.put("purpose_b2b", CardPurpose.B2B.value());
        params.put("now", Instant.now());

        @Language("SQL") String sql =
                "with subquery as (" +
                    "update cards set purpose = :purpose_b2b::card_purpose, updated_at = :now " +
                    "where uid = :uid and status = 'active'::card_status " +
                        "and purpose = :purpose_b2b_primary::card_purpose " +
                        "and external_id != :external_id" +
                ")" +
                "insert into cards (uid, purpose, external_id, " +
                "created_at, status, updated_at) values " +
                "(:uid, :purpose_b2b_primary::card_purpose, :external_id, " +
                ":now, 'active'::card_status, :now) " +
                "on conflict (uid, external_id) " +
                "do update set purpose = :purpose_b2b_primary::card_purpose, " +
                        "status = 'active'::card_status, updated_at = :now " +
                "returning *";

        return jdbcTemplate.query(sql, (rs, num) -> parseRow(rs), params).first();
    }

    @Override
    public Option<CardEntity> updatePurpose(PassportUid uid, String externalId, CardPurpose purpose) {
        if (purpose.equals(CardPurpose.B2B_PRIMARY)) {
            return Option.of(setB2BPrimaryCard(uid, externalId));
        }
        else {
            MapF<String, Object> params = Cf.hashMap();
            params.put("purpose", purpose.value());
            params.put("now", Instant.now());
            params.put("uid", uid.toString());
            params.put("external_id", externalId);

            return jdbcTemplate.queryForOption("update cards " +
                    "set purpose = :purpose::card_purpose, updated_at = :now " +
                    "where uid = :uid and external_id = :external_id " +
                    "returning *", (rs, num) -> parseRow(rs), params);
        }
    }

    @Override
    public void updateStatus(UUID id, CardStatus newStatus) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("status", newStatus.value());
        params.put("now", Instant.now());
        params.put("id", id);

        jdbcTemplate.update("update cards set status = :status::card_status, " +
                "updated_at = :now where id = :id", params);
    }

    @Override
    public Option<CardEntity> updateStatusAndPurpose(UUID id, CardStatus status, CardPurpose purpose) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("status", status.value());
        params.put("purpose", purpose.value());
        params.put("now", Instant.now());
        params.put("id", id);

        return jdbcTemplate.queryForOption("update cards set status = :status::card_status, " +
                "purpose = :purpose::card_purpose, updated_at = :now where id = :id " +
                "returning *", (rs, num) -> parseRow(rs), params);
    }

    @Override
    @Transactional
    public void updateStatus(ListF<String> externalIds, CardStatus newStatus) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("status", newStatus.value());
        params.put("now", Instant.now());
        params.put("external_ids", externalIds);

        jdbcTemplate.update("update cards set status = :status::card_status, " +
                "updated_at = :now where external_id in (:external_ids)", params);
    }

    private MapF<String, Object> makeParamsFromInsertData(InsertData insertData) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("uid", insertData.getUid().toString());
        params.put("purpose", insertData.getPurpose());
        params.put("external_id", insertData.getExternalId());
        params.put("status", insertData.getStatus());
        params.put("now", Instant.now());
        return params;
    }
}
