package ru.yandex.chemodan.app.psbilling.core.dao.cards.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.intellij.lang.annotations.Language;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.cards.TrustCardBindingDao;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardBindingStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.TrustCardBinding;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class TrustCardBindingDaoImpl extends AbstractDaoImpl<TrustCardBinding>
        implements TrustCardBindingDao {

    public TrustCardBindingDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "group_trust_card_bindings";
    }

    @Override
    public TrustCardBinding parseRow(ResultSet rs) throws SQLException {
        return new TrustCardBinding(UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                CardBindingStatus.R.fromValue(rs.getString("status")),
                PassportUid.cons(Long.parseLong(rs.getString("operator_uid"))),
                Option.ofNullable(rs.getString("transaction_id")),
                Option.ofNullable(rs.getString("error")),
                Option.ofNullable(rs.getString("card_id")).map(UUID::fromString));
    }

    @Override
    public TrustCardBinding insert(InsertData dataToInsert) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", now);
        params.put("status", dataToInsert.getStatus().value());
        params.put("operator_uid", dataToInsert.getOperatorUid().toString());
        params.put("transaction_id", dataToInsert.getTransactionId().orElse((String) null));

        @Language("SQL") String sql =
                "INSERT INTO group_trust_card_bindings (created_at, " +
                        "status, operator_uid, transaction_id) " +
                        "VALUES (:now, :status, :operator_uid, :transaction_id) " +
                        "RETURNING *";
        return jdbcTemplate.queryForOption(sql, (rs, i) -> parseRow(rs), params).get();
    }

    @Override
    public Option<TrustCardBinding> setTransactionId(UUID id, String transactionId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("id", id);
        params.put("transaction_id", transactionId);

        @Language("SQL")
        String sql = "UPDATE group_trust_card_bindings " +
                "SET transaction_id = :transaction_id " +
                "WHERE id = :id RETURNING *";
        return jdbcTemplate.queryForOption(sql, (rs, i) -> parseRow(rs), params);
    }

    @Override
    public Option<TrustCardBinding> updateStatusIfInNotSuccess(TrustCardBinding trustCardBinding) {
        MapF<String, Object> params = Cf.map("status_to_set", trustCardBinding.getStatus().value(),
                "status_to_check", CardBindingStatus.SUCCESS.value(),
                "id", trustCardBinding.getId());
        String condition = "";
        if (trustCardBinding.getError().isPresent()) {
            params.put("error", trustCardBinding.getError());
            condition += ", error = :error ";
        }
        if (trustCardBinding.getCardId().isPresent()) {
            params.put("card_id", trustCardBinding.getCardId());
            condition += ", card_id = :card_id ";
        }

        @Language("SQL")
        String sql = "UPDATE group_trust_card_bindings " +
                "SET status = :status_to_set " +
                condition +
                "WHERE id = :id AND status != :status_to_check RETURNING *";
        return jdbcTemplate.queryForOption(sql, (rs, i) -> parseRow(rs), params);
    }

    @Override
    public Option<TrustCardBinding> findByTransactionId(String transactionId) {
        MapF<String, Object> params = Cf.map("transaction_id", transactionId);

        @Language("SQL")
        String sql = "SELECT * FROM group_trust_card_bindings " +
                "WHERE transaction_id = :transaction_id LIMIT 1";
        return jdbcTemplate.queryForOption(sql, (rs, i) -> parseRow(rs), params);
    }

    @Override
    public ListF<TrustCardBinding> findInitBindings(Instant createdBefore,
                                                    int batchSize, Option<UUID> idFrom) {
        MapF<String, Object> params = Cf.map(
                "created_before", createdBefore,
                "batch_size", batchSize,
                "status", CardBindingStatus.INIT.value()
        );

        @Language("SQL")
        String query = "SELECT * FROM " + getTableName() +
                " WHERE created_at < :created_before " +
                " AND status = :status ";
        if (idFrom.isPresent()) {
            params = params.plus1("id", idFrom.get());
            query += "AND id > :id ";
        }
        query += "ORDER BY id LIMIT :batch_size";
        return jdbcTemplate.query(query, (rs, i) -> parseRow(rs), params);
    }

    @Override
    public int countStaleRecords(Duration duration) {
        return jdbcTemplate.queryForInt(
                "SELECT count(*) FROM " + getTableName() + " WHERE created_at < ? and status = ?",
                Instant.now().minus(duration),
                CardBindingStatus.INIT.value()
        );
    }
}
