package ru.yandex.chemodan.app.psbilling.core.dao.features.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;
import java.util.function.BiFunction;

import javax.annotation.Nonnull;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function2;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.features.ServiceFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.entities.features.FeatureWithOwner;
import ru.yandex.chemodan.app.psbilling.core.entities.features.IssuedFeature;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.ChildSynchronizableRecordJDBCHelper;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.SynchronizationStatus;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;
import ru.yandex.misc.time.TimeUtils;

@SuppressWarnings("SqlResolve")
public abstract class AbstractServiceFeatureDao
        <TFeature extends IssuedFeature<TOwner>, TOwner, TInsertData extends ServiceFeatureDao.InsertData<TOwner,
                TFeature>,
                TFeatureWithOwner extends FeatureWithOwner<TOwner>>
        extends AbstractDaoImpl<TFeature>
        implements ServiceFeatureDao<TFeature, TOwner, TInsertData, TFeatureWithOwner> {
    private final BiFunction<TOwner, UUID, TFeatureWithOwner> featureCreator;
    private final ChildSynchronizableRecordJDBCHelper<TFeature> childSynchronizableRecordJDBCHelper;

    private final String parentIdColumnName;
    private final String ownerIdColumnName;


    public AbstractServiceFeatureDao(JdbcTemplate3 jdbcTemplate, String parentIdColumnName,
                                     String ownerIdColumnName,
                                     Function2<TOwner, UUID, TFeatureWithOwner> featureCreator) {
        super(jdbcTemplate);
        this.featureCreator = featureCreator;
        this.childSynchronizableRecordJDBCHelper = new ChildSynchronizableRecordJDBCHelper<>(
                jdbcTemplate, getTableName(), Option.of(parentIdColumnName), this::parseRowTranslated);
        this.ownerIdColumnName = ownerIdColumnName;
        this.parentIdColumnName = parentIdColumnName;
    }

    @Override
    public ChildSynchronizableRecordJDBCHelper<TFeature> getChildSynchronizableRecordDaoHelper() {
        return childSynchronizableRecordJDBCHelper;
    }

    @Override
    public abstract String getTableName();

    @Override
    public void batchInsert(ListF<TInsertData> toInsert, Target target) {
        Instant now = Instant.now();
        jdbcTemplate.batchUpdate(
                "insert into " + getTableName() + " " +
                        "(created_at," + parentIdColumnName + ",product_feature_id, product_template_feature_id," + ownerIdColumnName + ",target," +
                        "target_updated_at,status,status_updated_at,actual_enabled_at,actual_disabled_at) " +
                        "values (?, ?, ?, ?, ?, ?, ?, ?, ?, null, null)",
                toInsert.map(m -> new Object[]{
                        now,
                        m.getParentServiceId(),
                        m.getProductFeatureId(),
                        m.getProductTemplateFeatureId(),
                        m.getOwnerId(),
                        target.value(),
                        now,
                        SynchronizationStatus.INIT,
                        now,
                })
        );
    }

    @Nonnull
    @Override
    public TFeature insert(TInsertData insertData, Target target) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("parent_service_id", insertData.getParentServiceId());
        params.put("product_feature_id", insertData.getProductFeatureId());
        params.put("product_template_feature_id", insertData.getProductTemplateFeatureId());
        params.put("target", target.value());
        params.put("status", SynchronizationStatus.INIT.value());
        params.put("ownerId", insertData.getOwnerId());
        params.put("now", now);

        return jdbcTemplate.query(
                "insert into " + getTableName() + " " +
                        "(created_at," + parentIdColumnName + ",product_feature_id, product_template_feature_id, " + ownerIdColumnName + ",target," +
                        "target_updated_at,status,status_updated_at,actual_enabled_at,actual_disabled_at) " +
                        "values" +
                        "(:now,:parent_service_id,:product_feature_id,:product_template_feature_id,:ownerId," +
                        ":target,:now,:status,:now,null,null) " +
                        "RETURNING *",
                (rs, num) -> parseRow(rs), params).first();
    }

    @Override
    public void setTargetState(ListF<UUID> ids, Target target) {
        if (ids.isEmpty()) {
            return;
        }

        MapF<String, Object> params = Cf.hashMap();
        params.put("new_target", target.value());
        params.put("sync_status", SynchronizationStatus.INIT.value());
        params.put("now", Instant.now());
        params.put("ids", ids);

        jdbcTemplate.update("update " + getTableName() + " set target = :new_target, status = :sync_status," +
                "status_updated_at = :now, target_updated_at = :now, next_try = null " +
                "where id in ( :ids )", params);

    }

    @Override
    public ListF<TFeatureWithOwner> findForSynchronization() {
        return jdbcTemplate.query(
                "select distinct psf." + ownerIdColumnName + ", pf.feature_id from " + getTableName() + " psf "
                        + "join product_features pf on psf.product_feature_id = pf.id "
                        + "where psf.status IN (:statuses_not_actual) "
                        + "and (psf.next_try is null or psf.next_try < :now)",
                (rs, i) -> featureCreator.apply(castOwnerId(rs.getObject(ownerIdColumnName)),
                        UUID.fromString(rs.getString("feature_id"))
                ),
                Cf.map("statuses_not_actual", Cf.x(SynchronizationStatus.values())
                                .filterNot(s -> s.equals(SynchronizationStatus.ACTUAL))
                                .map(SynchronizationStatus::value),
                        "now", Instant.now()
                )
        );
    }

    @Override
    public ListF<TFeatureWithOwner> findForSynchronization(UUID parentServiceId) {
        return jdbcTemplate.query(
                "select distinct psf." + ownerIdColumnName + ", pf.feature_id from " + getTableName() + " psf "
                        + "join product_features pf on psf.product_feature_id = pf.id "
                        + "where psf." + parentIdColumnName + " = :parent_service_id and psf.status IN " +
                        "(:statuses_not_actual) "
                        + "and (psf.next_try is null or psf.next_try < :now)",
                (rs, i) -> featureCreator.apply(
                        castOwnerId(rs.getObject(ownerIdColumnName)),
                        UUID.fromString(rs.getString("feature_id"))
                ),
                Cf.map("statuses_not_actual", Cf.x(SynchronizationStatus.values())
                                .filterNot(s -> s.equals(SynchronizationStatus.ACTUAL))
                                .map(SynchronizationStatus::value),
                        "parent_service_id", parentServiceId,
                        "now", Instant.now()
                )
        );
    }

    @Override
    public int countNotActualUpdatedBefore(Duration triggeredBefore) {
        return jdbcTemplate.queryForObject(
                "SELECT count(*) FROM " + getTableName() + " WHERE " +
                        "coalesce(next_try, status_updated_at) < ? and status <> ?",
                Integer.class,
                Instant.now().minus(triggeredBefore),
                SynchronizationStatus.ACTUAL.value()
        );
    }

    @Override
    public ListF<TFeature> findAndLockEnabledOrNotActual(
            TOwner ownerId, UUID featureId, UUID lockId, Duration lockTime) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("owner_id", ownerId);
        params.put("feature_id", featureId);
        params.put("status_actual", SynchronizationStatus.ACTUAL.value());
        params.put("target_enabled", Target.ENABLED.value());
        params.put("locked_till", now.plus(lockTime));
        params.put("lock_id", lockId);
        params.put("now", now);

        return jdbcTemplate.query(
                "with db_locked as ( " +
                        "         SELECT psf.* from " + getTableName() + " psf " +
                        "                       JOIN product_features pf ON psf.product_feature_id = pf.id " +
                        "                       WHERE psf." + ownerIdColumnName + " = :owner_id " +
                        "                       AND pf.feature_id = :feature_id " +
                        "                       AND (psf.target = :target_enabled OR psf.status <> :status_actual) " +
                        "         for update of psf), " +
                        "logical_lockable as ( " +
                        "    select * from db_locked where locked_till is null or locked_till < :now " +
                        "), " +
                        "all_lockable(all_lockable) as (select (select count(*) from db_locked) = (select count(*) " +
                        "from logical_lockable)) " +
                        "update " + getTableName() + " set locked_till = :locked_till, lock_id = :lock_id " +
                        "         from all_lockable " +
                        "         where all_lockable.all_lockable and id in (select id from db_locked) " +
                        "         returning * ",
                (rs, i) -> parseRow(rs),
                params
        );
    }

    @Override
    public void unlock(UUID lockId) {
        jdbcTemplate.update("update " + getTableName() + " set lock_id = null, locked_till = null where lock_id = ?",
                lockId);
    }

    @Override
    public void setStatusActualAndUnlock(UUID lockId, MapF<UUID, Target> idsAndTarget) {
        setStatusActualAndUnlockIfNeed(Option.of(lockId), idsAndTarget);
    }

    private void setStatusActualAndUnlockIfNeed(Option<UUID> lockId, MapF<UUID, Target> idsAndTarget) {
        String unlockSql = lockId.isPresent() ?
                ",lock_id = case when us.lock_id = input.lock_id then null else us.lock_id end "
                        + ",locked_till = case when us.lock_id = input.lock_id then null else us.locked_till end "
                : "";

        jdbcTemplate.batchUpdate(
                "with input(id, target, lock_id, now) as (select ?, ?, ?::UUID,  ?::timestamp) "
                        + "update " + getTableName() + " us set "
                        + "status = case when us.target = input.target then '" +
                        SynchronizationStatus.ACTUAL.value()
                        + "' else status end, "
                        + "actual_enabled_at = case when input.target = '" + Target.ENABLED.value()
                        + "' then input.now else actual_enabled_at end, "
                        + "actual_disabled_at = case when input.target = '" + Target.DISABLED.value()
                        + "' then input.now else actual_disabled_at end, "
                        + "status_updated_at = input.now, "
                        + "next_try = null "
                        + unlockSql
                        + "from input "
                        + "where us.id = input.id",
                idsAndTarget
                        .mapEntries((id, target) -> new Object[]{id, target.value(), lockId.orElse((UUID) null),
                                Instant.now()})
        );
    }

    @Override
    public void snoozeSynchronization(ListF<UUID> ids, Instant delayUntil) {
        if (ids.isEmpty()) {
            return;
        }

        MapF<String, Object> params = Cf.hashMap();
        params.put("ids", ids);
        params.put("delay_until", delayUntil);
        params.put("status", SynchronizationStatus.SNOOZING.value());
        params.put("now", Instant.now());

        jdbcTemplate.update("update " + getTableName() + " set status_updated_at = :now, status = :status ," +
                " next_try = :delay_until where id in ( :ids )", params);
    }

    @Override
    public void setStatusActual(UUID id, Target withTarget) {
        setStatusActualAndUnlockIfNeed(Option.empty(), Cf.map(id, withTarget));
    }

    @Override
    public TFeature parseRow(ResultSet rs) throws SQLException {
        //noinspection unchecked
        return create(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                Target.R.fromValue(rs.getString("target")),
                new Instant(rs.getTimestamp("status_updated_at")),
                SynchronizationStatus.R.fromValue(rs.getString("status")),
                new Instant(rs.getTimestamp("status_updated_at")),
                Option.ofNullable(rs.getTimestamp("actual_enabled_at")).map(Instant::new),
                Option.ofNullable(rs.getTimestamp("actual_disabled_at")).map(Instant::new),
                UUID.fromString(rs.getString(parentIdColumnName)),
                UUID.fromString(rs.getString("product_feature_id")),
                (TOwner) rs.getObject(ownerIdColumnName),
                Option.ofNullable(TimeUtils.getInstant(rs, "next_try")),
                Option.ofNullable(rs.getString("product_template_feature_id")).map(UUID::fromString));
    }

    public abstract TFeature create(UUID id, Instant createdAt, Target target, Instant targetUpdatedAt,
                                    SynchronizationStatus status, Instant statusUpdatedAt,
                                    Option<Instant> actualEnabledAt,
                                    Option<Instant> actualDisabledAt, UUID parentServiceId, UUID productFeatureId,
                                    TOwner ownerId,
                                    Option<Instant> nextTry, Option<UUID> productTemplateFeatureId) throws SQLException;


    private TOwner castOwnerId(Object ownerId) {
        try {
            //noinspection unchecked
            return (TOwner) ownerId;
        } catch (ClassCastException ex) {
            throw new IllegalArgumentException("unable to cast ownerId " + ownerId + " to owner type");
        }
    }
}
