package ru.yandex.chemodan.app.psbilling.core.dao.features.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Map;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.features.FeatureCallbackContextDao;
import ru.yandex.chemodan.app.psbilling.core.entities.features.FeatureCallbackContext;
import ru.yandex.chemodan.app.psbilling.core.util.JsonHelper;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class FeatureCallbackContextDaoImpl extends AbstractDaoImpl<FeatureCallbackContext> implements
        FeatureCallbackContextDao {
    public FeatureCallbackContextDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "feature_callback_context";
    }

    @Override
    public FeatureCallbackContext parseRow(ResultSet rs) throws SQLException {
        return new FeatureCallbackContext(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                Option.ofNullable(rs.getString("uid")),
                Option.ofNullable(rs.getString("group_id")).map(UUID::fromString),
                UUID.fromString(rs.getString("feature_id")),
                Option.ofNullable(rs.getString("data")).map(JsonHelper::parseJsonQuiet)
        );
    }

    @Override
    public FeatureCallbackContext insertOrUpdate(InsertData insertData) {
        if (insertData.getGroupId().isPresent() && insertData.getUid().isPresent()
                || (insertData.getGroupId().isEmpty() && insertData.getUid().isEmpty())) {
            throw new IllegalArgumentException("either uid ot group_id should be set");
        }

        MapF<String, Object> params = Cf.hashMap();
        params.put("now", Instant.now());
        params.put("feature_id", insertData.getFeatureId());


        if (insertData.getUid().isPresent()) {
            params.put("uid", insertData.getUid().get());
            return jdbcTemplate.query(
                    "INSERT INTO feature_callback_context (created_at, updated_at, uid, group_id, feature_id, data) "
                            + "VALUES (:now, :now, :uid, null, :feature_id, null) "
                            + "ON CONFLICT (uid, feature_id) DO UPDATE SET updated_at = :now, data = NULL "
                            + "RETURNING *",
                    (rs, i) -> parseRow(rs),
                    params
            ).first();
        } else {
            params.put("group_id", insertData.getGroupId().get());
            return jdbcTemplate.query(
                    "INSERT INTO feature_callback_context (created_at, updated_at, uid, group_id, feature_id, data) "
                            + "VALUES (:now, :now, null, :group_id, :feature_id, null) "
                            + "ON CONFLICT (group_id, feature_id) DO UPDATE SET updated_at = :now, data = NULL "
                            + "RETURNING *",
                    (rs, i) -> parseRow(rs),
                    params
            ).first();
        }
    }

    @Override
    public void updateData(UUID id, Map<String, Object> data) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("id", id);
        params.put("data", JsonHelper.asString(data));
        params.put("now", Instant.now());

        jdbcTemplate.update(
                "UPDATE feature_callback_context "
                        + "SET data = :data::jsonb, updated_at = :now "
                        + "WHERE id = :id",
                params
        );
    }

    @Override
    public Option<FeatureCallbackContext> findByUidAndFeature(String uid, UUID featureId) {
        return jdbcTemplate.queryForOption(
                "SELECT * FROM feature_callback_context WHERE uid = :uid AND feature_id = :feature_id",
                (rs, i) -> parseRow(rs),
                Cf.map(
                        "uid", uid,
                        "feature_id", featureId
                )
        );
    }

    @Override
    public Option<FeatureCallbackContext> findByGroupAndFeature(UUID groupId, UUID featureId) {
        return jdbcTemplate.queryForOption(
                "SELECT * FROM feature_callback_context WHERE group_id = :group_id AND feature_id = :feature_id",
                (rs, i) -> parseRow(rs),
                Cf.map(
                        "group_id", groupId,
                        "feature_id", featureId
                )
        );
    }

    @Override
    public void deleteByUidAndFeature(String uid, UUID featureId) {
        jdbcTemplate.update(
                "DELETE FROM feature_callback_context WHERE uid = :uid AND feature_id = :feature_id",
                Cf.map(
                        "uid", uid,
                        "feature_id", featureId
                )
        );
    }

    @Override
    public void deleteByGroupAndFeature(UUID groupId, UUID featureId) {
        jdbcTemplate.update(
                "DELETE FROM feature_callback_context WHERE group_id = :group_id AND feature_id = :feature_id",
                Cf.map(
                        "group_id", groupId,
                        "feature_id", featureId
                )
        );
    }
}
