package ru.yandex.chemodan.app.psbilling.core.dao.groups.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import lombok.AllArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.DistributionPlatformCalculationDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.CalculationStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.DistributionServiceTransactionCalculation;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

@AllArgsConstructor
public class DistributionPlatformCalculationTasksDaoImpl implements DistributionPlatformCalculationDao {
    protected final JdbcTemplate3 jdbcTemplate;
    private final String tableName = "distribution_platform_transaction_calculations";

    @Override
    public ListF<DistributionServiceTransactionCalculation> lock(LocalDate calcMonth) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("calc_month", calcMonth.withDayOfMonth(1));
        params.put("obsolete_status", CalculationStatus.OBSOLETE);

        return jdbcTemplate.query("select * from " + tableName + " " +
                " where calc_month = :calc_month and status <> :obsolete_status FOR UPDATE",
                (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public boolean updateStatus(UUID taskId, CollectionF<CalculationStatus> previousStatuses,
                                CalculationStatus newStatus) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("prev_st", previousStatuses.map(CalculationStatus::value));
        params.put("new_st", newStatus.value());
        params.put("now", Instant.now());
        params.put("job_id", taskId);

        return jdbcTemplate.update("update " + tableName + " " +
                "set status = :new_st , updated_at = :now " +
                "where task_id = :job_id and status in ( :prev_st ) ", params) > 0;
    }

    @Override
    public DistributionServiceTransactionCalculation initCalculation(LocalDate calcMonth, Long clientId,
                                                                     UUID jobId, String bazingaJobId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("job_id", jobId);
        params.put("client_id", clientId);
        params.put("now", Instant.now());
        params.put("calc_month", calcMonth.withDayOfMonth(1));
        params.put("status", CalculationStatus.STARTING);
        params.put("bazinga_job_id", bazingaJobId);

        return jdbcTemplate.query("insert into " + tableName + " " +
                "(calc_month, client_id, updated_at, task_id, status, bazinga_job_id) " +
                "values (:calc_month, :client_id, :now, :job_id, :status, :bazinga_job_id) " +
                "returning *", (rs, rowNum) -> parseRow(rs), params).first();
    }

    @Override
    public DistributionServiceTransactionCalculation find(UUID jobId) {
        return jdbcTemplate.query("select * from distribution_platform_transaction_calculations " +
                " where task_id = ?", (rs, rowNum) -> parseRow(rs), jobId).first();
    }

    @Override
    public void setCalculationObsolete(LocalDate calcMonth) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("calc_month", calcMonth.withDayOfMonth(1));
        params.put("now", Instant.now());
        params.put("obsolete_status", CalculationStatus.OBSOLETE);
        params.put("change_statuses", Cf.list(CalculationStatus.COMPLETED, CalculationStatus.STARTED,
                CalculationStatus.STARTING).map(CalculationStatus::value));

        jdbcTemplate.update("update " + tableName + " " +
                "set status = :obsolete_status , updated_at = :now " +
                "where calc_month = :calc_month and status in ( :change_statuses ) ", params);
    }

    public DistributionServiceTransactionCalculation parseRow(ResultSet rs) throws SQLException {
        return new DistributionServiceTransactionCalculation(
                LocalDate.fromDateFields(new Instant(rs.getDate("calc_month")).toDate()),
                rs.getLong("client_id"),
                new Instant(rs.getTimestamp("updated_at")),
                CalculationStatus.R.fromValue(rs.getString("status")),
                UUID.fromString(rs.getString("task_id")),
                StringUtils.trimToNull(rs.getString("bazinga_job_id")));
    }

}
