package ru.yandex.chemodan.app.psbilling.core.dao.groups.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Currency;
import java.util.UUID;

import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.DistributionPlatformTransactionsDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.DistributionPlatformTransactionEntity;
import ru.yandex.chemodan.app.psbilling.core.util.CurrencyUtils;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class DistributionPlatformTransactionsDaoImpl extends AbstractDaoImpl<DistributionPlatformTransactionEntity>
        implements DistributionPlatformTransactionsDao {
    private final String productId;

    public DistributionPlatformTransactionsDaoImpl(JdbcTemplate3 jdbcTemplate, String productId) {
        super(jdbcTemplate);
        this.productId = productId;
    }

    public String getTableName() {
        return "distribution_platform_transactions";
    }

    @Override
    public void batchInsert(CollectionF<InsertData> transactions) {
        jdbcTemplate.batchUpdate(
                String.format("insert into %s (calc_month, group_id, act_id, paid_user_count, amount, currency, clid," +
                        " created_at)" +
                        " ", getTableName())
                        + "VALUES (?, ?, ?, ?, ?, ?, ?, now())",
                transactions.map(m -> new Object[]{
                        m.getCalcMonth().withDayOfMonth(1),
                        m.getGroupId(),
                        m.getActId(),
                        m.getPaidUserCount(),
                        m.getAmount(),
                        m.getCurrency().getCurrencyCode(),
                        m.getClid()
                })
        );
    }

    @Override
    public ListF<DistributionPlatformTransactionEntity> findTransactions(LocalDate month,
                                                                         Option<UUID> idGreaterThan,
                                                                         int batchSize) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("calc_month", month.withDayOfMonth(1));
        params.put("batch_size", batchSize);

        String condition = "";
        if (idGreaterThan.isPresent()) {
            params.put("id_from", idGreaterThan.get());
            condition = " and tr.id > :id_from ";
        }

        String sql = "select tr.*" +
                " from " + getTableName() + " tr" +
                " where tr.calc_month = :calc_month " + condition + " order by tr.id limit :batch_size";
        return jdbcTemplate.query(sql,
                (rs, num) -> parseRow(rs), params);
    }

    public ListF<ExportRow> findExportRows(LocalDate month) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("calc_month", month.withDayOfMonth(1));

        String sql = "select " +
                "       dpt.calc_month, " +
                "       dpt.clid, " +
                "       dpt.currency, " +
                "       sum(dpt.amount)          as amount, " +
                "       sum(dpt.paid_user_count) as paid_user_count " +
                "from " + getTableName() + " dpt " +
                "where calc_month = :calc_month " +
                "group by dpt.calc_month, dpt.clid, dpt.currency";
        return jdbcTemplate.query(sql, (rs, num) -> parseExportRow(rs), params);
    }

    @Override
    public int removeCalculations(LocalDate month, int batchSize) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("calc_month", month.withDayOfMonth(1));
        params.put("batch_size", batchSize);

        return jdbcTemplate
                .update("delete from " + getTableName() + " where id in " +
                        "(select id" +
                        "     from " + getTableName() +
                        "     where calc_month = :calc_month" +
                        "     limit :batch_size )", params);
    }

    public DistributionPlatformTransactionEntity parseRow(ResultSet rs) throws SQLException {
        return new DistributionPlatformTransactionEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                LocalDate.fromDateFields(new Instant(rs.getDate("calc_month")).toDate()),
                UUID.fromString(rs.getString("group_id")),
                rs.getBigDecimal("paid_user_count"),
                rs.getBigDecimal("amount"),
                Currency.getInstance(rs.getString("currency")),
                rs.getString("clid"),
                rs.getString("act_id")
        );
    }

    public ExportRow parseExportRow(ResultSet rs) throws SQLException {
        return new ExportRow(
                // https://st.yandex-team.ru/CHEMODAN-82951#62442d58dadba84a4c78600a
                new LocalDate(rs.getTimestamp("calc_month")).plusMonths(1),
                rs.getString("clid"),
                rs.getBigDecimal("paid_user_count"),
                rs.getBigDecimal("amount"),
                CurrencyUtils.normalize(rs.getString("currency")),
                productId);
    }
}
