package ru.yandex.chemodan.app.psbilling.core.dao.groups.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.intellij.lang.annotations.Language;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.model.TimeIntervalCondition;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupPaymentType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.BalanceClientType;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.SynchronizationStatus;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.commune.json.jackson.ObjectMapperX;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.resultSet.ResultSetUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;
import ru.yandex.misc.time.InstantInterval;

public class GroupDaoImpl extends AbstractDaoImpl<Group> implements GroupDao {
    private static final ObjectMapperX mapper = new ObjectMapperX(new ObjectMapper());

    public GroupDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    /*
    use for testing purposes
     */
    private final boolean storePaymentInfoInJson = true;
    private final boolean storePaymentInfoExternal = true;

    @Override
    public Option<Group> findGroup(@Nonnull GroupType groupType, @Nonnull String externalId, @Nullable UUID parentGroupId) {
        if(groupType.isSubgroup() && parentGroupId == null)
            throw new IllegalArgumentException("Can't find for subgroup without parentGroupId");
        MapF<String, Object> params = Cf.hashMap();
        params.put("type", groupType.value());
        params.put("external_id", externalId);
        params.put("parent_group_id", parentGroupId);
        @Language("SQL")
        String query = "select * from groups where" +
                " group_type = :type" +
                " and group_external_id = :external_id" +
                " and parent_group_id " + (parentGroupId == null ? " is null" : " = :parent_group_id");
        return jdbcTemplate.queryForOption(query,
                (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    @Transactional
    public Group updatePaymentInfo(UUID groupId, BalancePaymentInfo paymentInfo) {
        Instant now = Instant.now();
        Group result;
        if (storePaymentInfoInJson) {
            MapF<String, Object> params = Cf.hashMap();
            params.put("payment_info", mapper.writeValueAsString(paymentInfo));
            params.put("id", groupId);
            params.put("now", now);

            result = jdbcTemplate.queryForOption("update groups set updated_at = :now, payment_info = " +
                    ":payment_info::json " +
                    " where id = :id returning *", (rs, rowNum) -> parseRow(rs), params).get();
        }
        if (storePaymentInfoExternal) {
            UUID paymentInfoId = upsertPaymentInfo(paymentInfo, now);
            MapF<String, Object> params = Cf.map(
                    "now", now,
                    "payment_info_id", paymentInfoId,
                    "group_id", groupId);
            @Language("SQL")
            String updatePaymentInfoIdQuery = "update groups set" +
                    " balance_payment_info_id = :payment_info_id, updated_at = :now" +
                    " where id = :group_id" +
                    " returning *";
            result = jdbcTemplate.queryForOption(updatePaymentInfoIdQuery, (rs, num) -> parseRow(rs), params).get();
        }
        return result;
    }


    @Override
    @Transactional
    public void setAutoBillingForClient(Long clientId, boolean b2bAutoBillingEnabled) {
        Instant now = Instant.now();

        MapF<String, Object> params = Cf.hashMap();
        params.put("client_id", clientId);
        params.put("now", now);
        params.put("auto_billing_enabled", b2bAutoBillingEnabled);
        @Language("SQL")
        String sql = "update balance_payment_infos set" +
                " updated_at = :now," +
                " b2b_auto_billing_enabled = :auto_billing_enabled" +
                " where client_id = :client_id";
        jdbcTemplate.update(sql, params);
        MapF<String, Object> fallbackParams = Cf.hashMap();
        fallbackParams.put("client_id", String.valueOf(clientId));
        fallbackParams.put("now", now);
        @Language("SQL")
        String fallback = "update groups set updated_at = :now, " +
                "payment_info = jsonb_set(payment_info, '{b2b_auto_billing_enabled}', '" + b2bAutoBillingEnabled +
                "', true) " +
                "where payment_info->>'client_id' = :client_id";
        jdbcTemplate.update(fallback, fallbackParams);

    }

    @Override
    public Group lockGroup(UUID groupId) {
        return jdbcTemplate
                .queryForObject("select * from groups where id = ? for update", (rs, num) -> parseRow(rs), groupId);
    }

    @Override
    public void updateMembersSynchronizationStatus(UUID groupId, boolean synchronizationStopped) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("id", groupId);
        params.put("now", Instant.now());
        params.put("members_synchronization_stopped", synchronizationStopped);

        jdbcTemplate.update("update groups set members_synchronization_stopped = :members_synchronization_stopped, " +
                "updated_at = :now where id = :id", params);
    }

    @Override
    public void updateStatus(UUID groupId, GroupStatus newStatus) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("id", groupId);
        params.put("now", Instant.now());
        params.put("status", newStatus.value());

        jdbcTemplate.update("update groups set updated_at = :now, status = :status " +
                " where id = :id and status <> :status", params);
    }

    @Nonnull
    @Override
    @Transactional
    public Group insert(InsertData dataToInsert) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("group_type", dataToInsert.getType().value());
        params.put("owner_uid", dataToInsert.getOwnerUid().getUid());
        params.put("group_external_id", dataToInsert.getExternalId());
        params.put("now", now);
        params.put("grace_period_hours", dataToInsert.getGracePeriod().getStandardHours());
        params.put("status_active", GroupStatus.ACTIVE.value());
        params.put("members_synchronization_stopped", dataToInsert.isSynchronizationStopped());
        params.put("clid", dataToInsert.getClid().orElse((String) null));
        params.put("parent_group_id", dataToInsert.getParentGroupId().orElse((UUID) null));
        UUID paymentInfoId;
        String paymentInfo;
        if (dataToInsert.getPaymentInfo() == null) {
            paymentInfoId = null;
            paymentInfo = null;
        } else {
            paymentInfoId = storePaymentInfoExternal ? upsertPaymentInfo(dataToInsert.getPaymentInfo(), now) : null;
            paymentInfo = storePaymentInfoInJson ? mapper.writeValueAsString(dataToInsert.getPaymentInfo()) : null;
        }
        params.put("payment_info_id", paymentInfoId);
        params.put("payment_info", paymentInfo);
        @Language("SQL")
        String query = "insert into groups (group_type, group_external_id, owner_uid, created_at,updated_at," +
                " payment_info, balance_payment_info_id, status, grace_period_hours, members_synchronization_stopped," +
                " " +
                "members_next_sync_dt, clid, parent_group_id)" +
                " values(:group_type, :group_external_id, :owner_uid, :now, :now, :payment_info::json, " +
                " :payment_info_id, :status_active, :grace_period_hours, :members_synchronization_stopped, :now," +
                "  :clid, :parent_group_id)" +
                " RETURNING *";
        return jdbcTemplate.query(query, (rs, num) -> parseRow(rs), params).first();
    }

    @Nonnull
    @Override
    public String getTableName() {
        return "groups";
    }

    @Override
    public Group parseRow(ResultSet rs) throws SQLException {
        String paymentInfoValue = rs.getString("payment_info");
        String paymentInfoId = rs.getString("balance_payment_info_id");
        Option<UUID> parentGroupId = Option.ofNullable(rs.getString("parent_group_id")).map(UUID::fromString);
        Option<BalancePaymentInfo> jsonPaymentInfo; //TODO remove after migration
        if (!StringUtils.isEmpty(paymentInfoValue)) {
            jsonPaymentInfo = Option.of(mapper.readValue(BalancePaymentInfo.class, paymentInfoValue));
        } else {
            jsonPaymentInfo = Option.empty();
        }
        String groupStatus = rs.getString("status");
        String gracePeriodStr = rs.getString("grace_period_hours");
        Option<Long> gracePeriodHours = org.apache.commons.lang3.StringUtils.isNumeric(gracePeriodStr) ?
                Option.of(Long.valueOf(gracePeriodStr)) : Option.empty();

        Function0<Option<BalancePaymentInfo>> paymentInfoProvider =
                () -> Option.of(findPaymentInfo(UUID.fromString(paymentInfoId)));
        UUID groupId = UUID.fromString(rs.getString("id"));
        return new Group(
                groupId,
                new Instant(rs.getTimestamp("created_at")),
                GroupType.R.fromValue(rs.getString("group_type")),
                rs.getString("group_external_id"),
                PassportUid.cons(rs.getLong("owner_uid")),
                paymentInfoId == null ? () -> jsonPaymentInfo : paymentInfoProvider.memoize(),
                new Instant(rs.getTimestamp("updated_at")),
                GroupStatus.R.fromValue(groupStatus),
                gracePeriodHours.map(Duration::standardHours),
                rs.getBoolean("members_synchronization_stopped"),
                new Instant(rs.getTimestamp("members_next_sync_dt")),
                Option.ofNullable(rs.getString("clid")),
                parentGroupId,
                ((Function0<Option<Group>>) () -> parentGroupId.map(this::findById)).memoize(),
                ((Function0<ListF<Group>>) () -> getChildGroups(groupId)).memoize());
    }

    private BalancePaymentInfo parsePaymentInfoRow(ResultSet rs) throws SQLException {
        return new BalancePaymentInfo(
                rs.getLong("client_id"),
                PassportUid.cons(rs.getLong("client_uid")),
                BalanceClientType.R.fromValue(rs.getString("client_type")),
                rs.getBoolean("b2b_auto_billing_enabled"));
    }

    @Override
    public SetF<String> findGroupProductCodesWithActiveServicesByExternalGroupIds(GroupType type,
                                                                                  CollectionF<String> groupExternalIds) {
        if (groupExternalIds.isEmpty()) {
            return Cf.set();
        }
        return jdbcTemplate.query(
                "SELECT gp.code FROM group_products gp "
                        + "JOIN group_services gs ON gp.id = gs.group_product_id "
                        + "JOIN groups g ON gs.group_id = g.id "
                        + "WHERE g.group_type = :group_type "
                        + "AND g.group_external_id IN (:group_external_ids) "
                        + "AND gs.status = :status_actual "
                        + "AND gs.target = :target_enabled ",
                ResultSetUtils.defaultSingleColumnRowMapper(String.class),
                Cf.map(
                        "group_type", type.value(),
                        "group_external_ids", groupExternalIds,
                        "target_enabled", Target.ENABLED.value(),
                        "status_actual", SynchronizationStatus.ACTUAL.value()
                )
        ).unique();
    }

    @Override
    public void insertAgreement(UUID groupId, UUID productOwnerId) {
        jdbcTemplate.update("insert into accepted_agreements (product_owner_id, group_id) values (?, ?) " +
                "on conflict do nothing", productOwnerId, groupId);
    }

    @Override
    @Transactional
    public void removePaymentInfo(UUID groupId) {
        @Language("SQL")
        String query = "update groups set payment_info = ? where id = ?";
        jdbcTemplate.update(query, null, groupId);
        query = "update groups set balance_payment_info_id = ? where id = ?";
        jdbcTemplate.update(query, null, groupId);
    }

    @Override
    public ListF<Group> filterGroupsByAcceptedAgreement(
            GroupType groupType, ListF<String> groupExternalIds, UUID productOwner) {
        if (groupExternalIds.isEmpty()) {
            return Cf.list();
        }
        @Language("SQL")
        String query = "SELECT g.* FROM groups g JOIN accepted_agreements aa ON aa.group_id = g.id " +
                "WHERE g.group_type = :group_type AND g.group_external_id IN (:external_ids) " +
                "AND aa.product_owner_id = :product_owner_id";
        return jdbcTemplate.query(
                query,
                (rs, rowNum) -> parseRow(rs), Cf.map(
                        "group_type", groupType,
                        "product_owner_id", productOwner,
                        "external_ids", groupExternalIds));
    }

    @Override
    public ListF<Group> getGroupsToCheckForUid(PassportUid uid, int batchSize, GroupType groupType,
                                               Option<UUID> greaterThan) {
        MapF<String, Object> params =
                getCommonParametersForBillingStatusCheck(batchSize, groupType);
        params.put("uid_json", mapper.writeValueAsString(Cf.map("uid", uid.toString())));
        params.put("uid", uid.toString());

        String groupIdCondition = "";
        if (greaterThan.isPresent()) {
            params.put("id", greaterThan.get());
            groupIdCondition += " and g.id > :id";
        }
        // @formatter:off
        @Language("SQL")
        String sql = "select * from groups g" +
                " where g.payment_info is not null " +
                " and g.payment_info->'uid' is not null " +
                " and g.payment_info @> :uid_json::jsonb " +
                " and g.group_type = :type " +
                groupIdCondition +
                " order by g.id limit :size";
        // @formatter:on
        return jdbcTemplate.query(sql, (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public ListF<PassportUid> findUidsToCheckBillingStatus(int batchSize, GroupType groupType,
                                                           Option<PassportUid> greaterThan) {
        MapF<String, Object> params = getCommonParametersForBillingStatusCheck(batchSize, groupType);

        String havingCondition = "";
        if (greaterThan.isPresent()) {
            params.put("uid", greaterThan.get().toString());
            havingCondition += " having g.payment_info->>'uid' > :uid";
        }

        @Language("SQL")
        String sql = "select g.payment_info->>'uid' AS uid from groups g where " +
                (" ( exists (select 1 from group_services gs " +
                        "         where gs.group_id = g.id " +
                        "            and gs.actual_disabled_at is null" +
                        "            and gs.b_date <= :now " +
                        "         limit 1)" +
                        " or g.status <> :status_active ) " +
                        " and g.payment_info is not null " +
                        " and g.payment_info->'uid' is not null " +
                        " and g.group_type = :type ") +
                " group by uid " +
                havingCondition +
                " order by uid limit :size";
        return jdbcTemplate.query(sql, (rs, rowNum) -> PassportUid.cons(Long.parseLong(rs.getString("uid"))), params);
    }

    @Override
    public ListF<Group> findGroups(ListF<UUID> groupIds) {
        if (groupIds.isEmpty()) {
            return Cf.list();
        }
        return jdbcTemplate.query("select * from " + getTableName() + " where id in (:ids)",
                (rs, rowNum) -> parseRow(rs), Cf.map("ids", groupIds));
    }

    @Override
    public Option<Group> getParentGroup(UUID childGroupId) {
        @Language("SQL")
        String query = "" +
                "select * from groups where id = (" +
                "   select parent_group_id from groups" +
                "   where id = :child_id" +
                ")";
        return jdbcTemplate.queryForOption(query, (rs, rowNum) -> parseRow(rs), Cf.map("child_id", childGroupId));
    }

    @Override
    public ListF<Group> getChildGroups(UUID parentGroupId) {
        @Language("SQL")
        String query = "select * from groups where parent_group_id = :parent_id";
        return jdbcTemplate.query(query, (rs, rowNum) -> parseRow(rs), Cf.map("parent_id", parentGroupId));
    }

    @Override
    public ListF<Group> findGroupsWithClid(int batchSize, GroupType groupType,
                                           Option<TimeIntervalCondition> billingDateIntervalConditionO,
                                           Option<UUID> greaterThan) {
        MapF<String, Object> params = getCommonParametersForBillingStatusCheck(batchSize, groupType);

        String greaterThanCondition;
        if (greaterThan.isPresent()) {
            params.put("uuid", greaterThan.get());
            greaterThanCondition = " and g.id > :uuid";
        } else {
            greaterThanCondition = "";
        }
        String dateCondition;
        if (billingDateIntervalConditionO.isPresent()) {
            TimeIntervalCondition condition = billingDateIntervalConditionO.get();
            params.put("time_lower_bound", condition.getLowerBound());
            params.put("time_upper_bound", condition.getUpperBound());
            dateCondition =
                    " and (created_at between :time_lower_bound and :time_upper_bound" +
                            "     or (created_at <= :time_lower_bound and target = 'enabled')" +
                            "     or (created_at <= :time_lower_bound and actual_disabled_at >= :time_lower_bound))";
        } else {
            dateCondition = "";
        }

        @Language("SQL")
        String sql = "select g.* " +
                "from groups g " +
                "where " +
                "    exists (" +
                "       select 1 from group_services gs " +
                "       where gs.group_id = g.id " +
                "          and skip_transactions_export = false " +
                dateCondition +
                "       limit 1)" +
                " and g.payment_info is not null " +
                " and g.payment_info->'uid' is not null " +
                " and g.group_type = :type " +
                " and g.clid is not null " +
                greaterThanCondition +
                " order by id limit :size";
        return jdbcTemplate.query(sql, (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public ListF<Group> findGroupsWithActivePrepaidService(
            InstantInterval balanceVoidInterval, Option<UUID> groupMinId, int batchSize
    ) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("after", balanceVoidInterval.getStart());
        params.put("before", balanceVoidInterval.getEnd());
        params.put("payment_type", GroupPaymentType.PREPAID);
        params.put("target_enabled", Target.ENABLED);
        params.put("limit", batchSize);

        @Language("SQL")
        String query = "select distinct g.* " +
                "from group_services gs " +
                "join group_products gp on gp.id = gs.group_product_id " +
                "join groups g on g.id = gs.group_id " +
                "join client_balance cb on g.payment_info ->> 'client_id' = cb.client_id::text " +
                "where cb.balance_void_at >= :after" +
                " and cb.balance_void_at < :before" +
                " and gs.target = :target_enabled" +
                " and gs.hidden = false" +
                " and gp.payment_type = :payment_type::payment_type" +
                " and gp.price > 0";

        if (groupMinId.isPresent()) {
            params.put("id", groupMinId.get());
            query += " and g.id > :id";
        }
        query += " order by g.id limit :limit";

        return jdbcTemplate.query(query, (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public Group findByPaymentRequestId(UUID requestId) {
        MapF<String, Object> params = Cf.map("request_id", requestId);
        return jdbcTemplate.queryForOption("SELECT g.* " +
                "FROM group_trust_payment_requests AS gtpr JOIN groups AS g ON g.id = gtpr.group_id " +
                "WHERE gtpr.id = :request_id", (rs, i) -> parseRow(rs), params).get();
    }

    @Override
    public ListF<Group> findGroupsByPaymentInfoUid(PassportUid uid) {
        MapF<String, Object> params = Cf.map("uid", uid.toString());
        @Language("SQL")
        String query = "SELECT * FROM groups" +
                " WHERE" +
                " payment_info IS NOT NULL" +
                " AND payment_info->'uid' IS NOT NULL" +
                " AND payment_info->>'uid' = :uid";

        return jdbcTemplate
                .query(query, (rs, i) -> parseRow(rs), params);
    }

    @Override
    public ListF<Group> findGroupsByPaymentInfoClient(Long clientId) {
        MapF<String, Object> params = Cf.map("client_id", clientId.toString());
        @Language("SQL")
        String query = "SELECT * FROM groups" +
                " WHERE" +
                " payment_info IS NOT NULL" +
                " AND payment_info->'client_id' IS NOT NULL" +
                " AND payment_info->>'client_id' = :client_id";
        return jdbcTemplate
                .query(query, (rs, i) -> parseRow(rs), params);
    }

    @Override
    public void updateGroupNextSyncTime(UUID groupId, Instant newNextSyncTime) {
        MapF<String, Object> params = Cf.map("id", groupId,
                "new_val", newNextSyncTime);

        jdbcTemplate.update("update groups set members_next_sync_dt = :new_val where id = :id ", params);
    }

    @Override
    public int mixpaidClientsCount() {
        MapF<String, Object> params = Cf.map(
                "prepaid_type", GroupPaymentType.PREPAID,
                "postpaid_type", GroupPaymentType.POSTPAID,
                "target_enabled", Target.ENABLED);

        @Language("SQL") String sql = "" +
                "select count(1) as cnt\n" +
                "from (\n" +
                "         select g.payment_info ->> 'client_id'\n" +
                "         from groups g\n" +
                "                  join group_services gs on g.id = gs.group_id\n" +
                "                  join group_products gp on gs.group_product_id = gp.id\n" +
                "         where gp.hidden = false " +
                "           and gs.target = :target_enabled" +
                "         group by g.payment_info ->> 'client_id'\n" +
                "         having sum(case when gp.payment_type = 'postpaid'::payment_type then 1 else 0 end) != 0\n" +
                "            and sum(case when gp.payment_type = 'prepaid'::payment_type then 1 else 0 end) != 0\n" +
                "     ) as mixedClients\n";

        return jdbcTemplate.query(sql, (rs, i) -> rs.getInt("cnt"), params).first();

    }

    @Override
    public boolean updateGroupNextSyncTimeIfNotChanged(UUID groupId, Instant oldNextSyncTime,
                                                       Instant newNextSyncTime) {
        MapF<String, Object> params = Cf.map("id", groupId,
                "old_val", oldNextSyncTime,
                "new_val", newNextSyncTime);

        return jdbcTemplate.update("update groups set members_next_sync_dt = :new_val where id = :id " +
                        "and date_trunc('milliseconds', members_next_sync_dt) = :old_val",
                params) > 0;
    }

    @Override
    public ListF<Group> findGroupsWithDisabledPrepaidGsAndNoActiveGs(
            InstantInterval balanceVoidInterval, Option<UUID> groupMinId, int batchSize
    ) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("start", balanceVoidInterval.getStart());
        params.put("end", balanceVoidInterval.getEnd());
        params.put("payment_type", GroupPaymentType.PREPAID);
        params.put("target_enabled", Target.ENABLED);
        params.put("target_disabled", Target.DISABLED);
        params.put("limit", batchSize);

        @Language("SQL")
        String query = "select distinct on (g.id) g.* " +
                "from group_services gs " +
                "join group_products gp on gp.id = gs.group_product_id " +
                "join groups g on g.id = gs.group_id " +
                "join client_balance cb on g.payment_info ->> 'client_id' = cb.client_id::text " +
                "left join group_services gs_other " +
                "on g.id = gs_other.group_id and gs_other.target = :target_enabled and gs.hidden = false " +
                "where cb.balance_void_at between :start and :end " +
                " and gs.target = :target_disabled" +
                " and gs.hidden = false" +
                " and gp.payment_type = :payment_type::payment_type" +
                " and gp.price > 0" +
                " and gs_other.id is null";

        if (groupMinId.isPresent()) {
            params.put("id", groupMinId.get());
            query += " and g.id > :id";
        }
        query += " order by g.id limit :limit";

        return jdbcTemplate.query(query, (rs, rowNum) -> parseRow(rs), params);
    }

    @Nonnull
    private BalancePaymentInfo findPaymentInfo(UUID paymentInfoId) {
        @Language("SQL")
        String query = "select * from balance_payment_infos where id = ?";
        return jdbcTemplate.queryForObject(query,
                (rs, rowNum) -> parsePaymentInfoRow(rs), paymentInfoId);
    }


    private MapF<String, Object> getCommonParametersForBillingStatusCheck(int batchSize, GroupType groupType) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("size", batchSize);
        params.put("enabled_status", Target.ENABLED.value());
        params.put("status_active", GroupStatus.ACTIVE.value());
        params.put("type", groupType.value());
        params.put("now", Instant.now());
        return params;
    }

    @Nonnull
    private UUID upsertPaymentInfo(BalancePaymentInfo paymentInfo, Instant now) {
        MapF<String, Object> params = paymentInfoToParamMap(paymentInfo);
        params.put("now", now);

        @Language("SQL")
        String upsertQuery = "insert into balance_payment_infos " +
                " (client_id, client_uid, created_at, updated_at, client_type, b2b_auto_billing_enabled) VALUES " +
                " (:client_id, :uid, :now, :now, :client_type::balance_client_type, :auto_billing_enabled)" +
                " on conflict (client_id) do update set " +
                " updated_at = excluded.updated_at," +
                " b2b_auto_billing_enabled = excluded.b2b_auto_billing_enabled," +
                " client_type = excluded.client_type" + //TODO do we really need it?
                " returning id";
        return UUID.fromString(jdbcTemplate.queryForOption(upsertQuery, (rs, rowNum) -> rs.getString(1), params).get());
    }

    @NotNull
    private MapF<String, Object> paymentInfoToParamMap(BalancePaymentInfo paymentInfo) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("uid", paymentInfo.getPassportUid().toString());
        params.put("auto_billing_enabled", paymentInfo.isB2bAutoBillingEnabled());
        params.put("client_id", paymentInfo.getClientId());
        params.put("client_type", paymentInfo.getClientType());
        return params;
    }
}
