package ru.yandex.chemodan.app.psbilling.core.dao.groups.impl;

import java.math.BigDecimal;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Currency;
import java.util.UUID;

import javax.annotation.Nonnull;

import org.apache.commons.lang3.StringUtils;
import org.intellij.lang.annotations.Language;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupProductDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupPaymentType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupProductEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupProductType;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProduct;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;
import ru.yandex.misc.time.TimeUtils;


public class GroupProductDaoImpl extends AbstractDaoImpl<GroupProductEntity> implements GroupProductDao {

    public GroupProductDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public Option<GroupProductEntity> findProductByCode(@Nonnull String code) {
        return jdbcTemplate.queryForOption("select * from group_products where code = ?",
                (rs, rowNum) -> parseRow(rs), code);
    }

    @Override
    public ListF<GroupProductEntity> findByProductCodes(ListF<String> codes) {
        if (codes.isEmpty()) {
            return Cf.list();
        }
        return jdbcTemplate.query("select * from group_products where code in ( :codes )",
                (rs, rowNum) -> parseRow(rs), Cf.map("codes", codes));
    }

    @Override
    public GroupProductEntity findByGroupServiceId(UUID groupServiceId) {
        return jdbcTemplate.queryForOption("select gp.* from group_products gp " +
                        " join group_services gs on gp.id = gs.group_product_id " +
                        " where gs.id = ?",
                (rs, rowNum) -> parseRow(rs), groupServiceId).get();
    }

    @Override
    public ListF<GroupProductEntity> findAvailableAddons(UUID mainProductId) {
        @Language("SQL")
        String query = "select gp.* from group_products gp " +
                " join group_product_addons gpa on gp.id = gpa.addon_product_id " +
                " where gpa.enabled = true" +
                " and gpa.main_product_id = ?";
        return jdbcTemplate.query(query,
                (rs, rowNum) -> parseRow(rs), mainProductId);
    }

    @Override
    public ListF<GroupProductEntity> findEligibleMainProducts(UUID addonProductId) {
        @Language("SQL")
        String query = "select gp.* from group_products gp " +
                " join group_product_addons gpa on gp.id = gpa.main_product_id " +
                " where gpa.enabled = true" +
                " and gpa.addon_product_id = ?";
        return jdbcTemplate.query(query,
                (rs, rowNum) -> parseRow(rs), addonProductId);
    }

    @Override
    public ListF<GroupProductEntity> findByIds(CollectionF<UUID> productIds) {
        if (productIds.isEmpty()) {
            return Cf.list();
        }
        return jdbcTemplate.query("select * from group_products where id = ANY (ARRAY[ :ids ])",
                (rs, rowNum) -> parseRow(rs), Cf.map("ids", productIds));
    }

    @Override
    public ListF<GroupProductEntity> findByProductLine(UUID lineId) {
        return jdbcTemplate.query("select g.* from " +
                        " group_products g join group_products_to_product_lines p_l on p_l.group_product_id = g.id " +
                        " where p_l.product_line_id  = :id" +
                        " order by p_l.order_num ",
                (rs, num) -> parseRow(rs)
                , Cf.map("id", lineId));
    }

    @Nonnull
    @Override
    public GroupProductEntity insert(InsertData dataToInsert) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("code", dataToInsert.getCode());
        params.put("now", now);
        params.put("hidden", dataToInsert.isHidden());
        params.put("user_product_id", dataToInsert.getUserProduct());
        params.put("balance_product_name", dataToInsert.getBalanceProductName().orElse((String) null));
        params.put("price", dataToInsert.getPricePerUserInMonth());
        params.put("display_original_price", dataToInsert.getDisplayOriginalPrice().orElse((BigDecimal) null));
        params.put("currency", dataToInsert.getPriceCurrency().getCurrencyCode());
        params.put("singleton", dataToInsert.isSingleton());
        params.put("skip_transactions_export", dataToInsert.isSkipTransactionsExport());
        params.put("trial_definition_id", dataToInsert.getTrialDefinitionId().orElse((UUID) null));
        params.put("available_to", dataToInsert.getAvailableTo().orElse((Instant) null));
        params.put("title_tanker_key_id", dataToInsert.getTitleTankerKeyId().orElse((UUID) null));
        params.put("best_offer", dataToInsert.isBestOffer());
        params.put("payment_type", dataToInsert.getPaymentType());
        params.put("product_type", dataToInsert.getProductType());
        params.put("display_discount_percent", dataToInsert.getDisplayDiscountPercent().orElse((BigDecimal) null));

        @Language("SQL")
        String query = "insert into group_products (code,user_product_id, created_at,updated_at, balance_product_name," +
                " price, display_original_price, currency, singleton, skip_transactions_export, " +
                "trial_definition_id,available_to, hidden," +
                "title_tanker_key_id, payment_type, product_type, display_discount_percent) " +
                "values(:code, :user_product_id, :now, :now, :balance_product_name, :price, " +
                ":display_original_price," +
                " :currency, :singleton, :skip_transactions_export, :trial_definition_id, :available_to, " +
                ":hidden, " +
                ":title_tanker_key_id, :payment_type::payment_type, :product_type::group_product_type, " +
                ":display_discount_percent) " +
                "RETURNING *";
        return jdbcTemplate.queryForOption(query, (rs, rowNum) -> parseRow(rs), params).get();
    }

    @Override
    public String getTableName() {
        return "group_products";
    }

    @Override
    public GroupProductEntity parseRow(ResultSet rs) throws SQLException {
        String trialDefinitionId = rs.getString("trial_definition_id");

        return new GroupProductEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                rs.getString("code"),
                UUID.fromString(rs.getString("user_product_id")),
                Option.ofNullable(StringUtils.trimToNull(rs.getString("balance_product_name"))),
                rs.getBigDecimal("price"),
                Currency.getInstance(rs.getString("currency")),
                rs.getBoolean("singleton"),
                rs.getBoolean("skip_transactions_export"),
                StringUtils.isBlank(trialDefinitionId) ? Option.empty() : Option.of(UUID.fromString(trialDefinitionId)),
                Option.ofNullable(TimeUtils.getInstant(rs, "available_to")),
                rs.getBoolean("hidden"),
                Option.ofNullable(rs.getString("title_tanker_key_id")).map(UUID::fromString),
                rs.getBoolean("is_best_offer"),
                GroupPaymentType.R.valueOf(rs.getString("payment_type")),
                Option.ofNullable(rs.getBigDecimal("display_original_price")),
                GroupProductType.R.valueOf(rs.getString("product_type")),
                Option.ofNullable(rs.getBigDecimal("display_discount_percent")));
    }

    @Override
    public void linkAddon(GroupProduct main, GroupProduct addon) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", now);
        params.put("main_id", main.getId());
        params.put("addon_id", addon.getId());
        params.put("enabled", true);
        @Language("SQL")
        String query = "insert into group_product_addons (main_product_id, addon_product_id, enabled, created_at, updated_at) values " +
                " (:main_id, :addon_id, :enabled, :now, :now) " +
                " on conflict (main_product_id, addon_product_id) do update set " +
                "   enabled = excluded.enabled," +
                "   updated_at = excluded.updated_at" +
                " where group_product_addons.enabled != excluded.enabled"; //here is false-warning, but I check it works find

        jdbcTemplate.update(query, params);
    }

    @Override
    public void unlinkAddon(GroupProduct main, GroupProduct addon) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", now);
        params.put("main_id", main.getId());
        params.put("addon_id", addon.getId());
        params.put("enabled", false);
        @Language("SQL")
        String query = "update group_product_addons " +
                " set enabled = :enabled," +
                " updated_at = :now" +
                " where main_product_id = :main_id" +
                " and addon_product_id = :addon_id" +
                " and enabled != :enabled";

        jdbcTemplate.update(query, params);
    }
}
