package ru.yandex.chemodan.app.psbilling.core.dao.groups.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Currency;
import java.util.UUID;

import org.intellij.lang.annotations.Language;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Interval;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.entities.AbstractEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupPaymentType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.PriceOverrideReason;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.IParentSynchronizableRecordJdbcHelper;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.ParentSynchronizableRecordJdbcHelper2;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.SynchronizationStatus;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.util.date.DateTimeUtils;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;
import ru.yandex.misc.time.InstantInterval;

import static ru.yandex.bolts.collection.Tuple2.tuple;

public class GroupServiceDaoImpl extends AbstractDaoImpl<GroupService>
        implements GroupServiceDao {
    private final IParentSynchronizableRecordJdbcHelper<GroupService> parentSynchronizableRecordJdbcHelper;
    private static final Logger logger = LoggerFactory.getLogger(GroupServiceDaoImpl.class);

    public GroupServiceDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
        this.parentSynchronizableRecordJdbcHelper = new ParentSynchronizableRecordJdbcHelper2<>(
                jdbcTemplate, getTableName(), this::parseRowTranslated,
                "group_service_members", "group_service_features",
                (pAlias, cAlias) -> pAlias + ".id = " + cAlias + ".group_service_id");
    }

    @Override
    public IParentSynchronizableRecordJdbcHelper<GroupService> getParentSynchronizableRecordDaoHelper() {
        return parentSynchronizableRecordJdbcHelper;
    }

    @Override
    public ListF<GroupService> find(UUID groupId, UUID productId) {
        return jdbcTemplate
                .query("select * from group_services where group_id = ? and group_product_id = ?",
                        (rs, rowNum) -> parseRow(rs), groupId, productId);
    }

    @Override
    public ListF<GroupService> findGroupServicesToSync(UUID groupId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("target_enabled", Target.ENABLED.value());
        params.put("target_disabled", Target.DISABLED.value());
        params.put("status_init", SynchronizationStatus.INIT.value());
        params.put("group_id", groupId);

        //use CREATE INDEX group_services__group_id__target__status__ix ON group_services (group_id, target, status)
        return jdbcTemplate.query(
                "select * from group_services where group_id = :group_id and " +
                        "( target IN (:target_enabled) or (target = :target_disabled and status = :status_init))",
                (rs, i) -> parseRow(rs), params
        );
    }

    @Override
    public ListF<GroupService> findActiveGroupServicesByPaymentInfoClient(Long clientId, Currency currency) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("target_enabled", Target.ENABLED.value());
        params.put("clientId", clientId.toString());
        params.put("currency", currency.getCurrencyCode());

        @Language("SQL")
        String query =
                "select gs.* from groups g" +
                " join group_services gs on gs.group_id = g.id" +
                " join group_products gp on gp.id = gs.group_product_id" +
                " where g.payment_info is not null" +
                " and g.payment_info->>'client_id'= :clientId " +
                " and target = :target_enabled" +
                " and gp.currency = :currency";
        return jdbcTemplate.query(
                query,
                (rs, i) -> parseRow(rs), params);
    }

    @Override
    public ListF<UUID> groupIdsToSync() {
        MapF<String, Object> params = Cf.hashMap();
        params.put("target_enabled", Target.ENABLED.value());
        params.put("target_disabled", Target.DISABLED.value());
        params.put("status_init", SynchronizationStatus.INIT.value());
        params.put("now", Instant.now());

        //use CREATE INDEX group_services__group_id__target__status__ix ON group_services (group_id, target, status)
        return jdbcTemplate.query(
                "select distinct gs.group_id " +
                        " from group_services gs" +
                        "    join groups g on g.id = gs.group_id" +
                        "    left join group_service_features gsf on gs.id = gsf.group_service_id" +
                        " where (gs.target IN (:target_enabled)" +
                        "           or (gs.target = :target_disabled and gs.status = :status_init))" +
                        " and ((g.members_synchronization_stopped = false and g.members_next_sync_dt <= :now)" +
                        "      or gsf.status = :status_init)",
                (rs, i) -> UUID.fromString(rs.getString("group_id")), params
        );
    }

    @Override
    public ListF<GroupService> find(UUID groupId, Target... targets) {
        return find(groupId, false, targets);
    }

    @Override
    public ListF<GroupService> find(UUID groupId, boolean withSubgroups, Target... targets) {
        return find(Cf.list(groupId), withSubgroups, targets);
    }

    @Override
    public ListF<GroupService> find(ListF<UUID> groupIds, boolean withSubgroups, Target... targets) {
        if (groupIds.isEmpty()) {
            return Cf.list();
        }

        MapF<String, Object> params = Cf.map(
                "groupIds", groupIds,
                "statuses", Cf.list(targets).map(Target::value));
        @Language("SQL")
        String subgroupCondition = "select distinct id from groups" +
                " where parent_group_id in (:groupIds)" +
                " union distinct " +
                " select id from groups where id in (:groupIds)";

        String groupIdCondition = withSubgroups ? subgroupCondition : ":groupIds";
        @Language("SQL")
        String query = "select * from group_services" +
                " where group_id in (" + groupIdCondition + ")" +
                (targets.length > 0 ? "and target in ( :statuses )" : "");
        return jdbcTemplate
                .query(query,
                        (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public MapF<GroupService, ListF<Interval>> getEnabledUserServicesInDayPaidIntervals(LocalDate date,
                                                                                        Page page) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("start_uid", page.getMin());
        params.put("end_uid", page.getMax());

        @Language("SQL")
        String sql = getEnabledUserServicesInDayPaidIntervalsCondition(" and id >= :start_uid and id <= :end_uid ");
        return getEnabledUserServicesInDayPaidIntervals(date, sql, params);
    }

    @Override
    public MapF<GroupService, ListF<Interval>> getEnabledUserServicesInDayPaidIntervals(LocalDate date,
                                                                                        ListF<GroupService> groupServices) {
        if (groupServices.size() == 0) {
            return Cf.map();
        }

        MapF<String, Object> params = Cf.hashMap();
        params.put("gs_ids", groupServices.map(AbstractEntity::getId));
        @Language("SQL")
        String sql = getEnabledUserServicesInDayPaidIntervalsCondition(" and id in (:gs_ids) ");
        return getEnabledUserServicesInDayPaidIntervals(date, sql, params);
    }

    @Override
    public void updateNextBillingDate(UUID groupServiceId, Option<Instant> nextBillingDateO) {
        logger.info("update bdate for service {} to {}", groupServiceId, nextBillingDateO);
        MapF<String, Object> params = Cf.hashMap();
        params.put("groupServiceId", groupServiceId);
        params.put("b_date", nextBillingDateO.orElse((Instant) null));

        jdbcTemplate.update("update group_services set b_date = :b_date " +
                " where id = :groupServiceId", params);
    }

    @Override
    public void updatePrepaidNextBillingDateForClient(Long clientId, Currency currency,
                                                      Option<Instant> nextBillingDateO) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("clientId", clientId.toString());
        params.put("currency", currency.getCurrencyCode());
        params.put("bDate", nextBillingDateO.orElse((Instant) null));
        params.put("now", Instant.now());
        params.put("target_enabled", Target.ENABLED.value());
        params.put("prepaid_type", GroupPaymentType.PREPAID);

        @Language("SQL")
        String query =
                " update group_services set b_date = :bDate " +
                " where id in (" +
                "   select gs.id " +
                "   from group_services gs " +
                "      join group_products gp on gp.id = gs.group_product_id " +
                "      join groups g on gs.group_id = g.id " +
                "   where gs.target = :target_enabled " +
                "      and gp.payment_type = :prepaid_type::payment_type " +
                "      and gp.price > 0 " +
                "      and payment_info is not null and payment_info->>'client_id' = :clientId" +
                ")";
        jdbcTemplate.update(query, params);
    }

    @Override
    public ListF<GroupService> findGroupServices(ListF<UUID> groupServiceIds) {
        if (groupServiceIds.isEmpty()) {
            return Cf.list();
        }
        return jdbcTemplate.query("select * from " + getTableName() + " where id in (:ids)",
                (rs, rowNum) -> parseRow(rs), Cf.map("ids", groupServiceIds));
    }

    @Override
    public ListF<GroupService> findActiveGroupServicesByPaymentType(ListF<UUID> groupIds,
                                                                    GroupPaymentType paymentType) {
        if (groupIds.isEmpty()) {
            return Cf.list();
        }
        MapF<String, Object> params = Cf.hashMap();
        params.put("group_ids", groupIds);
        params.put("target_enabled", Target.ENABLED.value());
        params.put("payment_type", paymentType);

        return jdbcTemplate.query("" +
                        "select gs.* from " + getTableName() + " gs " +
                        "join group_products gp on gp.id = gs.group_product_id " +
                        "where gs.group_id IN (:group_ids) and " +
                        "gs.target = :target_enabled and " +
                        "gp.payment_type = :payment_type::payment_type",
                (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public ListF<GroupService> findGroupServicesWithNotHiddenTrialOverrideEndDateInBetween(Instant afterInclusive,
                                                                                           Instant beforeExclusive,
                                                                                           int batchSize,
                                                                                           Option<UUID> groupServiceIdMin) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("after", afterInclusive);
        params.put("before", beforeExclusive);
        params.put("limit", batchSize);
        params.put("reason", PriceOverrideReason.TRIAL.value());
        params.put("hidden", Boolean.FALSE);

        @Language("SQL")
        String query = "SELECT gs.* " +
                "FROM group_services AS gs JOIN group_service_price_overrides AS gspo ON gs.id = gspo" +
                ".group_service_id " +
                "WHERE gspo.end_date >= :after AND gspo.end_date < :before AND gspo.reason = :reason AND gspo.hidden " +
                "= :hidden ";
        if (groupServiceIdMin.isPresent()) {
            params.put("id", groupServiceIdMin.get());
            query += "AND gs.id > :id ";
        }
        query += "ORDER BY gs.id LIMIT :limit";
        return jdbcTemplate.query(query, (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public ListF<GroupService> findActiveGroupServicesCreatedInBetween(InstantInterval interval,
                                                                       Option<UUID> groupIdMin, int batchSize) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("start", interval.getStart());
        params.put("end", interval.getEnd());
        params.put("limit", batchSize);
        params.put("target_enabled", Target.ENABLED.value());

        @Language("SQL")
        String query = "SELECT distinct on (group_id) * " +
                "FROM group_services " +
                "WHERE created_at between :start AND :end AND hidden = false and target = :target_enabled ";
        if (groupIdMin.isPresent()) {
            params.put("group_id", groupIdMin.get());
            query += "AND group_id > :group_id ";
        }
        query += "ORDER BY group_id LIMIT :limit";
        return jdbcTemplate.query(query, (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public int countNotActualUpdatedBefore(Duration triggeredBefore) {
        return GroupServiceDao.super.countNotActualUpdatedBefore(triggeredBefore);
    }

    @Override
    public int getMistakenlyActiveServiceCount(Duration threshold) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("balanceVoidStartDate", Instant.now().minus(threshold));
        params.put("prepaidType", GroupPaymentType.PREPAID);

        @Language("SQL")
        String query = "select count(1) as cnt\n" +
                "from groups g\n" +
                "         join client_balance cb on g.payment_info ->> 'client_id' = cb.client_id::text\n" +
                "         join group_services gs on g.id = gs.group_id\n" +
                "         join group_products gp on gs.group_product_id = gp.id and gp.currency = cb" +
                ".balance_currency\n" +
                "where gs.target = 'enabled'\n" +
                "  and gs.hidden = false\n" +
                "  and gs.skip_transactions_export = false\n" +
                "  and cb.balance_void_at <= :balanceVoidStartDate\n" +
                "  and gp.payment_type = :prepaidType::payment_type";

        return jdbcTemplate.query(query, (rs, rowNum) -> rs.getInt("cnt"), params).get(0);
    }

    @Override
    public ListF<GroupService> findLastDisabledPrepaidServices(long clientId, Currency currency) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("target_enabled", Target.ENABLED.value());
        params.put("clientId", clientId);
        params.put("currency", currency.getCurrencyCode());
        params.put("target_disabled", Target.DISABLED);
        params.put("prepaid_type", GroupPaymentType.PREPAID);

        @Language("SQL")
        String sql = "select gs.*, g.id as g_id, gp.id as gp_id, cb.id as cb_id\n" +
                "from client_balance cb\n" +
                "         join groups g on g.payment_info ->> 'client_id' = cb.client_id::text\n" +
                "         join group_services gs on g.id = gs.group_id\n" +
                "         join group_products gp on gs.group_product_id = gp.id and gp.currency = cb.balance_currency\n" +
                "where cb.client_id = :clientId\n" +
                "  and cb.balance_amount <= 0\n" +
                "  and cb.balance_currency = :currency\n" +
                "  and gs.target = :target_disabled\n" +
                "  and gs.b_date = cb.balance_void_at\n" +
                "  and gp.payment_type = :prepaid_type::payment_type\n" +
                "  and gp.price > 0\n" +
                "  and gs.skip_transactions_export = false\n";
        return jdbcTemplate.query(
                sql,
                (rs, i) -> parseRow(rs), params);
    }

    @Override
    public GroupService updateSkipTransactionsExport(UUID groupServiceId, boolean skipTransactionsExport) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("id", groupServiceId);
        params.put("v", skipTransactionsExport);

        return jdbcTemplate.queryForOption("update group_services set skip_transactions_export = :v where id = :id" +
                " RETURNING *", (rs, num) -> parseRow(rs), params).get();
    }

    @Override
    public Option<Page> nextPageOfActiveServicesOnDay(LocalDate date, Option<UUID> greaterThan,
                                                      int batchSize) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("start_date", date);
        params.put("end_date", date.plusDays(1));
        params.put("batch_size", batchSize);

        String condition = "";
        if (greaterThan.isPresent()) {
            condition = "  and id > :min_id ";
            params.put("min_id", greaterThan.get());
        }
        @Language("SQL")
        String sql = "select max(t.id::TEXT) as max_id, min(t.id::TEXT) as min_id from (" +
                "select id from group_services " +
                " where created_at < :end_date " +
                "  and (actual_disabled_at is null or actual_disabled_at > :start_date) " +
                condition +
                "  order by id " +
                " limit :batch_size ) as t";
        Tuple2<String, String> tuple =
                jdbcTemplate.query(sql, (rs, num) -> tuple(rs.getString("min_id"), rs.getString("max_id")), params)
                        .first();
        if (StringUtils.isBlank(tuple.get1()) || StringUtils.isBlank(tuple.get2())) {
            return Option.empty();
        }

        return Option.of(new Page(UUID.fromString(tuple._1), UUID.fromString(tuple._2)));
    }

    @Override
    public ListF<GroupService> findActiveServicesOnDayForClient(LocalDate date, Long clientId) {

        MapF<String, Object> params = Cf.hashMap();
        params.put("date", date);
        params.put("clientId", clientId.toString());

        @Language("SQL")
        String sql =
                "select gs.* from group_services gs" +
                        " join groups g on g.id = gs.group_id" +
                        " where gs.created_at::date <= :date " +
                        "  and (actual_disabled_at is null or actual_disabled_at > :date) " +
                        "  and g.payment_info->>'client_id' = :clientId";
        return jdbcTemplate.query(sql, (rs, num) -> parseRow(rs), params);
    }


    @Override
    public boolean setTargetToDisabled(UUID groupServiceId, Target oldState) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("new_state", Target.DISABLED.value());
        params.put("old_state", oldState.value());
        params.put("status", SynchronizationStatus.INIT.value());
        params.put("id", groupServiceId);
        params.put("now", Instant.now());

        return jdbcTemplate.update(
                "update group_services set target = :new_state, target_updated_at = :now, " +
                        "status = :status, status_updated_at = :now " +
                        " where target = :old_state and id = :id", params) > 0;
    }

    @NotNull
    @Override
    public GroupService insert(InsertData dataToInsert) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("group_id", dataToInsert.getGroupId());
        params.put("product_id", dataToInsert.getProductId());
        params.put("deduplication_key", dataToInsert.getDeduplicationKey());
        params.put("tgt_state", dataToInsert.getTarget().value());
        params.put("sync_state", SynchronizationStatus.INIT.value());
        params.put("b_date", dataToInsert.getNextBDate().orElse((Instant) null));
        params.put("now", now);
        params.put("hidden", dataToInsert.isHidden());
        params.put("skip_transactions_export", dataToInsert.isSkipTransactionsExport());

        return jdbcTemplate.queryForOption(
                "insert into group_services " +
                        "(group_id, group_product_id, deduplication_key, b_date, created_at, updated_at, hidden," +
                        " target, target_updated_at, status, status_updated_at, actual_enabled_at, " +
                        "actual_disabled_at, skip_transactions_export) " +
                        " values " +
                        "(:group_id, :product_id, :deduplication_key, :b_date, :now, :now, :hidden, " +
                        " :tgt_state, :now, :sync_state, :now, null, null, :skip_transactions_export)" +
                        "RETURNING *",
                (rs, rowNum) -> parseRow(rs), params).get();
    }

    @Override
    public String getTableName() {
        return "group_services";
    }

    private GroupService parseRowSafe(ResultSet rs) {
        try {
            return parseRow(rs);
        } catch (SQLException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    @Override
    public void updateFirstFeatureDisabledAt(ListF<UUID> ids) {
        if (ids.isEmpty()) {
            return;
        }

        jdbcTemplate.update(
                "update group_services set first_feature_disabled_at = :now " +
                        "where id in ( :ids ) and first_feature_disabled_at is null and target = :target",
                Cf.map(
                        "now", Instant.now(),
                        "ids", ids,
                        "target", Target.DISABLED.value()
                )
        );
    }

    public GroupService parseRow(ResultSet rs) throws SQLException {
        Option<Timestamp> actualEnabled = Option.ofNullable(rs.getTimestamp("actual_enabled_at"));
        Option<Timestamp> actualDisabled = Option.ofNullable(rs.getTimestamp("actual_disabled_at"));
        Option<Timestamp> bDate = Option.ofNullable(rs.getTimestamp("b_date"));

        return new GroupService(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                // field should be not null, but unable to update all records at once
                new Instant(Option.ofNullable(rs.getTimestamp("updated_at"))
                        .orElse(rs.getTimestamp("status_updated_at"))),
                Target.R.fromValue(rs.getString("target")),
                new Instant(rs.getTimestamp("target_updated_at")),
                SynchronizationStatus.R.fromValue(rs.getString("status")),
                new Instant(rs.getTimestamp("status_updated_at")),
                actualEnabled.map(Instant::new),
                actualDisabled.map(Instant::new),
                UUID.fromString(rs.getString("group_id")),
                UUID.fromString(rs.getString("group_product_id")),
                bDate.map(Instant::new),
                Option.ofNullable(rs.getString("deduplication_key")),
                Option.ofNullable(rs.getTimestamp("first_feature_disabled_at")).map(Instant::new),
                rs.getBoolean("skip_transactions_export"),
                rs.getBoolean("hidden")
        );
    }

    private String getEnabledUserServicesInDayPaidIntervalsCondition(String idCondition) {
        return "with gs as ( " +
                "    select * " +
                "    from group_services " +
                "    where created_at <= :end_date " +
                "      and (actual_disabled_at is null " +
                "        or actual_disabled_at >= :start_date) " +
                idCondition +
                "      and not skip_transactions_export " +
                ")" +
                "select gs.*," +
                "       GREATEST(us.actual_enabled_at, :start_date)                         as start_date, " +
                "       LEAST(coalesce(us.first_feature_disabled_at, :end_date), :end_date) as end_date " +
                "from gs " +
                "         join group_service_members gsm on gsm.group_service_id = gs.id " +
                "         join user_services us on us.id = gsm.user_service_id " +
                "where us.actual_enabled_at <= :end_date " +
                "  AND (us.first_feature_disabled_at is null " +
                "    or us.first_feature_disabled_at >= :start_date) ";
    }


    private MapF<GroupService, ListF<Interval>> getEnabledUserServicesInDayPaidIntervals(
            LocalDate date, @Language("SQL") String sql, MapF<String, Object> params) {
        params.put("start_date", DateTimeUtils.toInstant(date));
        params.put("end_date", DateTimeUtils.toInstant(date.plusDays(1)));

        MapF<UUID, GroupService> servicesMap = Cf.hashMap();
        MapF<GroupService, ListF<Interval>> servicesIntervals = Cf.hashMap();

        jdbcTemplate.query(sql, rs -> {
            UUID groupServiceId = UUID.fromString(rs.getString("id"));
            GroupService groupService = servicesMap.getOrElseUpdate(groupServiceId, () -> parseRowSafe(rs));
            ListF<Interval> serviceIntervals = servicesIntervals.getOrElseUpdate(groupService, Cf::arrayList);
            Instant startDate = new Instant(rs.getTimestamp("start_date"));
            Instant endDate = new Instant(rs.getTimestamp("end_date"));
            if (endDate.isAfterNow()) {
                endDate = Instant.now();
            }
            if (endDate.isAfter(startDate)) {
                serviceIntervals.add(new Interval(startDate, endDate));
            }
        }, params);

        logger.info("got intervals: {}", servicesIntervals);
        return servicesIntervals;
    }
}
