package ru.yandex.chemodan.app.psbilling.core.dao.groups.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceMemberDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupServiceMember;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.ChildSynchronizableRecordJDBCHelper;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.ParentSynchronizableRecordJdbcHelper;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.SynchronizationStatus;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class GroupServiceMemberDaoImpl extends AbstractDaoImpl<GroupServiceMember> implements GroupServiceMemberDao {
    private final ChildSynchronizableRecordJDBCHelper<GroupServiceMember> childSynchronizableRecordJDBCHelper;
    private final ParentSynchronizableRecordJdbcHelper<GroupServiceMember> parentSynchronizableRecordJDBCHelper;

    public GroupServiceMemberDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
        this.childSynchronizableRecordJDBCHelper = new ChildSynchronizableRecordJDBCHelper<>(
                jdbcTemplate, getTableName(), Option.of("group_service_id"), this::parseRowTranslated);
        this.parentSynchronizableRecordJDBCHelper = new ParentSynchronizableRecordJdbcHelper<>(
                jdbcTemplate, getTableName(), this::parseRowTranslated, "user_services",
                (pAlias, cAlias) -> pAlias + ".user_service_id = " + cAlias + ".id");
    }


    @Override
    public ParentSynchronizableRecordJdbcHelper<GroupServiceMember> getParentSynchronizableRecordDaoHelper() {
        return parentSynchronizableRecordJDBCHelper;
    }

    @Override
    public void updateUserServiceId(UUID id, UUID userServiceId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("id", id);
        params.put("user_service_id", userServiceId);

        jdbcTemplate.update("update group_service_members set user_service_id = :user_service_id " +
                "where id = :id", params);
    }

    @Override
    public int getEnabledMembersCount(UUID groupServiceId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("groupServiceId", groupServiceId);
        params.put("targetEnabled", Target.ENABLED.value());
        return jdbcTemplate.query("select count(1) from group_service_members gsm" +
                        " where gsm.group_service_id = :groupServiceId" +
                        " and gsm.target = :targetEnabled",
                (rs, i) -> rs.getInt(1), params).first();
    }

    @Override
    public ChildSynchronizableRecordJDBCHelper<GroupServiceMember> getChildSynchronizableRecordDaoHelper() {
        return childSynchronizableRecordJDBCHelper;
    }

    @Override
    public void batchInsert(ListF<InsertData> toInsert, Target target) {
        Instant now = Instant.now();
        jdbcTemplate.batchUpdate(
                "insert into group_service_members(group_service_id, uid, created_at, " +
                        "target, target_updated_at, status, status_updated_at, actual_enabled_at, actual_disabled_at) "
                        + "VALUES (?, ?, ?, ?, ?, ?, ?, null, null)",
                toInsert.map(m -> new Object[]{
                        m.getGroupServiceId(),
                        m.getUid(),
                        now,
                        target.value(),
                        now,
                        SynchronizationStatus.INIT.value(),
                        now
                })
        );
    }

    @Override
    public String getTableName() {
        return "group_service_members";
    }


    @Override
    public GroupServiceMember parseRow(ResultSet rs) throws SQLException {
        Option<Timestamp> actualEnabled = Option.ofNullable(rs.getTimestamp("actual_enabled_at"));
        Option<Timestamp> actualDisabled = Option.ofNullable(rs.getTimestamp("actual_disabled_at"));
        String userServiceId = rs.getString("user_service_id");

        return new GroupServiceMember(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                Target.R.fromValue(rs.getString("target")),
                new Instant(rs.getTimestamp("target_updated_at")),
                SynchronizationStatus.R.fromValue(rs.getString("status")),
                new Instant(rs.getTimestamp("status_updated_at")),
                actualEnabled.map(Instant::new),
                actualDisabled.map(Instant::new),
                UUID.fromString(rs.getString("group_service_id")),
                rs.getString("uid"),
                userServiceId == null ? Option.empty() : Option.of(UUID.fromString(userServiceId))
        );
    }
}
