package ru.yandex.chemodan.app.psbilling.core.dao.groups.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServicePriceOverrideDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupServicePriceOverride;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.PriceOverrideReason;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;
import ru.yandex.misc.time.TimeUtils;

public class GroupServicePriceOverrideDaoImpl extends AbstractDaoImpl<GroupServicePriceOverride> implements
        GroupServicePriceOverrideDao
{

    public GroupServicePriceOverrideDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "group_service_price_overrides";
    }

    @Override
    public GroupServicePriceOverride parseRow(ResultSet rs) throws SQLException {
        String trialUsageId = rs.getString("trial_usage_id");
        return new GroupServicePriceOverride(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                UUID.fromString(rs.getString("group_service_id")),
                rs.getBigDecimal("price"),
                new Instant(rs.getTimestamp("start_date")),
                Option.ofNullable(TimeUtils.getInstant(rs, "end_date")),
                StringUtils.isBlank(trialUsageId) ? Option.empty() : Option.of(UUID.fromString(trialUsageId)),
                PriceOverrideReason.R.fromValue(rs.getString("reason")),
                rs.getBoolean("hidden")
        );
    }

    @Override
    public GroupServicePriceOverride insert(InsertData data) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", Instant.now());
        params.put("group_service_id", data.getGroupServiceId());
        params.put("price", data.getPricePerUserInMonth());
        params.put("start_date", data.getStartDate());
        params.put("end_date", data.getEndDate() == null ? null : data.getEndDate().getOrNull());
        params.put("trial_usage_id", data.getTrialUsageId() == null ? null : data.getTrialUsageId().getOrNull());
        params.put("reason", data.getReason().value());
        params.put("hidden", data.isHidden());

        return jdbcTemplate.queryForOption(
                "insert into group_service_price_overrides " +
                        "(group_service_id,created_at,price,start_date,end_date,trial_usage_id,reason, hidden) " +
                        " values " +
                        "(:group_service_id, :now, :price, :start_date,:end_date, :trial_usage_id,:reason, :hidden)" +
                        "RETURNING *",
                (rs, rowNum) -> parseRow(rs), params).get();

    }

    @Override
    public MapF<UUID, ListF<GroupServicePriceOverride>> findByGroupServices(ListF<UUID> groupServiceIds) {
        if (groupServiceIds.isEmpty()) {
            return Cf.map();
        }
        return jdbcTemplate.query("select * from group_service_price_overrides where group_service_id in ( :ids )",
                (rs, num) -> parseRow(rs), Cf.map("ids", groupServiceIds))
                .groupBy(GroupServicePriceOverride::getGroupServiceId);
    }
}
