package ru.yandex.chemodan.app.psbilling.core.dao.groups.impl;

import java.sql.ResultSet;
import java.sql.SQLException;

import lombok.AllArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceTransactionCalculationDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.CalculationStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupServiceTransactionCalculation;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

@AllArgsConstructor
public class GroupServiceTransactionCalculationDaoImpl implements GroupServiceTransactionCalculationDao {
    protected final JdbcTemplate3 jdbcTemplate;

    @Override
    public GroupServiceTransactionCalculation lock(LocalDate date) {
        return jdbcTemplate.queryForObject("select * from group_service_transaction_calculations " +
                " where billing_date = ? FOR UPDATE", (rs, rowNum) -> parseRow(rs), date);
    }

    @Override
    public void insertIfAbsent(LocalDate date) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("billing_date", date);
        params.put("updated_at", Instant.now());
        params.put("status_starting", CalculationStatus.STARTING);

        jdbcTemplate.update(
                "insert into group_service_transaction_calculations " +
                        "(billing_date, updated_at, status, bazinga_calculation_job_id) " +
                        " values " +
                        "(:billing_date, :updated_at, :status_starting, null) " +
                        " ON CONFLICT DO NOTHING", params);
    }

    @Override
    public boolean updateStatus(LocalDate date, CollectionF<CalculationStatus> previousStatuses,
            CalculationStatus newStatus)
    {
        MapF<String, Object> params = Cf.hashMap();
        params.put("prev_st", previousStatuses.map(CalculationStatus::value));
        params.put("new_st", newStatus.value());
        params.put("now", Instant.now());
        params.put("b_date", date);

        return jdbcTemplate.update("update group_service_transaction_calculations " +
                "set status = :new_st , updated_at = :now " +
                "where billing_date = :b_date and status in ( :prev_st ) ", params) > 0;
    }

    @Override
    public LocalDate findLatestCalculation() {
        return jdbcTemplate.queryForObject(
                "select max(billing_date) as value from group_service_transaction_calculations",
                (rs, num) -> LocalDate.fromDateFields(new Instant(rs.getDate("value")).toDate()));
    }

    @Override
    public void saveBazingaJobId(LocalDate date, String jobId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("jobId", jobId);
        params.put("now", Instant.now());
        params.put("b_date", date);

        jdbcTemplate.update("update group_service_transaction_calculations " +
                "set bazinga_calculation_job_id = :jobId , updated_at = :now " +
                "where billing_date = :b_date", params);
    }

    public GroupServiceTransactionCalculation parseRow(ResultSet rs) throws SQLException {
        String bazingaCalculationJobId = rs.getString("bazinga_calculation_job_id");

        return new GroupServiceTransactionCalculation(
                LocalDate.fromDateFields(new Instant(rs.getDate("billing_date")).toDate()),
                new Instant(rs.getTimestamp("updated_at")),
                CalculationStatus.R.fromValue(rs.getString("status")),
                Option.ofNullable(StringUtils.trimToNull(bazingaCalculationJobId))
        );
    }

}
