package ru.yandex.chemodan.app.psbilling.core.dao.groups.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Currency;
import java.util.UUID;

import lombok.AllArgsConstructor;
import org.intellij.lang.annotations.Language;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceTransactionsDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupServiceTransaction;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

@AllArgsConstructor
public class GroupServiceTransactionsDaoImpl implements GroupServiceTransactionsDao {
    protected final JdbcTemplate3 jdbcTemplate;

    @Override
    public void batchInsert(CollectionF<GroupServiceTransaction> transactions) {
        jdbcTemplate.batchUpdate(
                "insert into group_service_transactions(billing_date, group_service_id, amount, currency, " +
                        "user_seconds_count, calculated_at) " +
                        "VALUES (?, ?, ?, ?, ?, ?)" +
                        "on conflict (billing_date, group_service_id) " +
                        "  do update set " +
                        "  amount = ?, currency = ?, user_seconds_count = ?, calculated_at = ?",
                transactions.map(m -> new Object[]{
                        m.getBillingDate(),
                        m.getGroupServiceId(),
                        m.getAmount(),
                        m.getCurrency().getCurrencyCode(),
                        m.getUserSecondsCount(),
                        m.getCalculatedAt(),
                        m.getAmount(),
                        m.getCurrency().getCurrencyCode(),
                        m.getUserSecondsCount(),
                        m.getCalculatedAt()
                })
        );
    }

    @Override
    public ListF<ExportRow> findTransactions(LocalDate date,
                                             Option<UUID> groupServiceGreaterThan, int batchSize) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("b_date", date);
        params.put("batch_size", batchSize);

        String condition = "";
        if (groupServiceGreaterThan.isPresent()) {
            params.put("gs_uuid", groupServiceGreaterThan.get());
            condition = " and tr.group_service_id > :gs_uuid ";
        }

        @Language("SQL")
        String sql = "select g.payment_info::json->>'client_id' as client_id, gp.balance_product_name, tr.*" +
                " from group_service_transactions tr " +
                " join group_services gs on gs.id = tr.group_service_id " +
                " join groups g on gs.group_id = g.id " +
                " join group_products gp on gp.id = gs.group_product_id " +
                " where tr.billing_date = :b_date " + condition + " order by tr.group_service_id limit :batch_size ";

        return jdbcTemplate.query(sql,
                (rs, num) -> new ExportRow(rs.getString("client_id"), rs.getString("balance_product_name"),
                        parseRow(rs)), params);
    }

    @Override
    public ListF<GroupServiceTransaction> findGroupTransactions(LocalDate month, UUID groupId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("b_date_from", month.withDayOfMonth(1));
        params.put("b_date_to", month.withDayOfMonth(1).plusMonths(1).minusDays(1));
        params.put("group_id", groupId);
        @Language("SQL")
        String sql = "select tr.*" +
                " from group_service_transactions tr " +
                " join group_services gs on gs.id = tr.group_service_id " +
                " where tr.billing_date between :b_date_from and :b_date_to and gs.group_id = :group_id";

        return jdbcTemplate.query(sql, (rs, num) -> parseRow(rs), params);
    }

    @Override
    public ListF<GroupServiceTransaction> findClientTransactionsFrom(LocalDate date, Long clientId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("b_date_from", date);
        params.put("clientId", clientId.toString());
        @Language("SQL")
        String sql = "select tr.*" +
                " from groups g" +
                " join group_services gs on gs.group_id = g.id " +
                " join group_service_transactions tr on gs.id = tr.group_service_id" +
                " where tr.billing_date >= :b_date_from " +
                "   and g.payment_info is not null and g.payment_info ->> 'client_id' = :clientId";

        return jdbcTemplate.query(sql, (rs, num) -> parseRow(rs), params);
    }

    @Override
    public ListF<Long> findTransactionsClients(LocalDate billingDate, Option<Long> from, int batchSize) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("billingDate", billingDate);
        params.put("batchSize", batchSize);
        String orderCondition = "";
        if (from.isPresent()) {
            params.put("fromClientId", from.get().toString());
            orderCondition = "   and g.payment_info ->> 'client_id' > :fromClientId";
        }


        @Language("SQL") String sql =
                "select distinct g.payment_info ->> 'client_id' as clientId" +
                        " from group_service_transactions tr" +
                        " join group_services gs on gs.id = tr.group_service_id" +
                        " join groups g on g.id = gs.group_id" +
                        " where tr.billing_date = :billingDate " +
                        "   and g.payment_info is not null" +
                        orderCondition +
                        " order by clientId" +
                        " limit :batchSize";
        return jdbcTemplate.query(sql, (rs, num) -> rs.getLong("clientId"), params);
    }

    @Override
    public int removeCalculations(LocalDate date, int batchSize) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("b_date", date);
        params.put("batch_size", batchSize);

        return jdbcTemplate
                .update("delete from group_service_transactions where (billing_date, group_service_id) in " +
                        "(select billing_date, group_service_id" +
                        "     from group_service_transactions" +
                        "     where billing_date = :b_date" +
                        "     limit :batch_size )", params);
    }

    private GroupServiceTransaction parseRow(ResultSet rs) throws SQLException {
        return new GroupServiceTransaction(
                LocalDate.fromDateFields(new Instant(rs.getDate("billing_date")).toDate()),
                UUID.fromString(rs.getString("group_service_id")),
                rs.getBigDecimal("amount"),
                Currency.getInstance(rs.getString("currency")),
                rs.getBigDecimal("user_seconds_count"),
                new Instant(rs.getTimestamp("calculated_at")));
    }
}
