package ru.yandex.chemodan.app.psbilling.core.dao.groups.impl;

import java.math.BigDecimal;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.intellij.lang.annotations.Language;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.Money;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupTrustPaymentRequestDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupTrustPaymentGroupServiceInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupTrustPaymentRequest;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.PaymentInitiationType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.PaymentRequestStatus;
import ru.yandex.commune.json.jackson.ObjectMapperX;
import ru.yandex.misc.db.resultSet.ResultSetExtras;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class GroupTrustPaymentRequestDaoImpl extends AbstractDaoImpl<GroupTrustPaymentRequest>
        implements GroupTrustPaymentRequestDao {
    private static final ObjectMapperX mapper = new ObjectMapperX(new ObjectMapper());

    public GroupTrustPaymentRequestDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "group_trust_payment_requests";
    }

    @Override
    public GroupTrustPaymentRequest parseRow(ResultSet rs) throws SQLException {
        String groupServicesInfoS = rs.getString("group_services_info");
        Option<GroupTrustPaymentGroupServiceInfo> groupServicesInfo =
                StringUtils.isEmpty(groupServicesInfoS) ? Option.empty()
                        : Option.of(mapper.readValue(GroupTrustPaymentGroupServiceInfo.class, groupServicesInfoS));

        return new GroupTrustPaymentRequest(UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                rs.getString("request_id"),
                PaymentRequestStatus.R.fromValue(rs.getString("status")),
                Option.ofNullable(rs.getString("error")),
                Option.ofNullable(rs.getString("transaction_id")),
                rs.getString("operator_uid"),

                // cleanup nullable fallback on CHEMODAN-82069
                Option.ofNullable(rs.getString("initiation_type"))
                        .map(PaymentInitiationType.R::fromValue).orElse(PaymentInitiationType.USER),
                // cleanup nullable fallback on CHEMODAN-82069
                new ResultSetExtras(rs).getLongO("client_id").orElse(-1L),

                Option.ofNullable(rs.getString("payment_period_coefficient")).map(Double::parseDouble),
                groupServicesInfo,
                Option.ofNullable(rs.getString("card_id")).map(UUID::fromString),
                getMoney(rs));
    }

    private Option<Money> getMoney(ResultSet rs) throws SQLException {
        Option<String> currency = Option.ofNullable(rs.getString("currency"));
        Option<BigDecimal> amount = Option.ofNullable(rs.getBigDecimal("amount"));
        if (amount.isPresent() && currency.isPresent()) {
            return Option.of(new Money(amount.get(), currency.get()));
        }
        return Option.empty();
    }

    @Override
    public GroupTrustPaymentRequest insert(InsertData dataToInsert) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", now);
        params.put("request_id", dataToInsert.getRequestId());
        params.put("status", dataToInsert.getStatus().value());
        params.put("transaction_id", dataToInsert.getTransactionId().orElse((String) null));
        params.put("operator_uid", dataToInsert.getOperatorUid());
        params.put("client_id", dataToInsert.getClientId());
        params.put("initiation_type", dataToInsert.getPaymentInitiationType());
        params.put("group_services_info", dataToInsert.getGroupServicesInfo().map(mapper::writeValueAsString)
                .orElse((String) null));
        params.put("payment_period_coefficient", dataToInsert.getPaymentCoefficient().orElse((Double) null));
        params.put("card_id", dataToInsert.getCardId().orElse((UUID) null));
        params.put("amount", dataToInsert.getMoney().getAmount());
        params.put("currency", dataToInsert.getMoney().getCurrency().getCurrencyCode());
        params.put("error", dataToInsert.getError().orElse((String) null));

        @Language("SQL") String sql =
                "INSERT INTO group_trust_payment_requests (created_at, request_id, " +
                        "status, transaction_id, operator_uid, initiation_type, client_id, " +
                        "payment_period_coefficient, " +
                        "card_id, group_services_info, amount, currency, error) " +
                        "VALUES (:now, :request_id, :status, :transaction_id, :operator_uid, " +
                        ":initiation_type::payment_initiation_type, :client_id, :payment_period_coefficient, " +
                        ":card_id, :group_services_info::jsonb, :amount, :currency, :error) " +
                        "RETURNING *";
        return jdbcTemplate.queryForOption(sql, (rs, i) -> parseRow(rs), params).get();
    }

    @Override
    public Option<GroupTrustPaymentRequest> updateStatusIfInInit(GroupTrustPaymentRequest groupTrustPaymentRequest) {
        MapF<String, Object> params = Cf.map("status_to_set", groupTrustPaymentRequest.getStatus().value(),
                "status_to_check", PaymentRequestStatus.INIT.value(),
                "id", groupTrustPaymentRequest.getId(),
                "error", groupTrustPaymentRequest.getError());
        String condition = "";
        if (groupTrustPaymentRequest.getError().isPresent()) {
            params.put("error", groupTrustPaymentRequest.getError());
            condition = ", error = :error ";
        }

        @Language("SQL")
        String sql = "UPDATE group_trust_payment_requests " +
                "SET status = :status_to_set " +
                condition +
                "WHERE id = :id AND status = :status_to_check RETURNING *";
        return jdbcTemplate.queryForOption(sql, (rs, i) -> parseRow(rs), params);
    }

    @Override
    public Option<GroupTrustPaymentRequest> findByRequestId(String requestId) {
        return jdbcTemplate.queryForOption("SELECT * FROM group_trust_payment_requests WHERE request_id = :request_id",
                (rs, i) -> parseRow(rs),
                Cf.map("request_id", requestId));
    }

    @Override
    public ListF<GroupTrustPaymentRequest> findInitPaymentsOlderThan(Instant olderThan, int batchSize,
                                                                     Option<UUID> idFrom) {
        MapF<String, Object> params = Cf.map(
                "older_than", olderThan,
                "batch_size", batchSize,
                "status", PaymentRequestStatus.INIT.value()
        );
        @Language("SQL")
        String query = "SELECT * FROM " + getTableName() + " WHERE created_at < :older_than AND status = :status ";
        if (idFrom.isPresent()) {
            params = params.plus1("id", idFrom.get());
            query += "AND id > :id ";
        }
        query += "ORDER BY id LIMIT :batch_size";
        return jdbcTemplate.query(query, (rs, i) -> parseRow(rs), params);
    }

    @Override
    public ListF<GroupTrustPaymentRequest> findRecentPayments(Long clientId, Option<Instant> from) {
        MapF<String, Object> params = Cf.map("clientId", clientId);

        @Language("SQL")
        String query = "SELECT * FROM group_trust_payment_requests " +
                "WHERE client_id = :clientId";
        if (from.isPresent()) {
            params = params.plus1("threshold", from.get());
            query += " and created_at >= :threshold";
        }
        return jdbcTemplate.query(query, (rs, i) -> parseRow(rs), params);
    }

    @Override
    public Option<GroupTrustPaymentRequest> updateTransactionIdIfNull(UUID id, String transactionId) {
        MapF<String, Object> params = Cf.map(
                "id", id,
                "transaction_id", transactionId
        );

        @Language("SQL")
        String sql = "UPDATE group_trust_payment_requests " +
                " SET transaction_id = :transaction_id" +
                " WHERE id = :id " +
                " AND transaction_id IS NULL " +
                " RETURNING *";
        return jdbcTemplate.queryForOption(sql, (rs, i) -> parseRow(rs), params);
    }

    @Override
    public BigDecimal getRecentAutoPaymentsSum(Instant from, PaymentRequestStatus status) {
        return getRecentAutoPaymentsSum(from, status, true);
    }

    @Override
    public BigDecimal getRecentResurrectPaymentsSum(Instant from, PaymentRequestStatus status) {
        return getRecentAutoPaymentsSum(from, status, false);
    }

    private BigDecimal getRecentAutoPaymentsSum(Instant from, PaymentRequestStatus status,
                                                boolean isGroupServicesInfoNull) {
        MapF<String, Object> params = Cf.map(
                "from", from,
                "status", status.value(),
                "init_type", PaymentInitiationType.AUTO
        );
        @Language("SQL")
        String sql =
                "select sum(pr.amount) as money\n" +
                        "from group_trust_payment_requests pr\n" +
                        "where pr.status = :status\n" +
                        "  and pr.initiation_type = :init_type::payment_initiation_type\n" +
                        (isGroupServicesInfoNull
                                ? "  and pr.group_services_info is null\n"
                                : "  and pr.group_services_info is not null\n") +
                        "  and pr.created_at > :from\n";

        return jdbcTemplate.queryForOption(sql,
                (rs, i) -> Option.ofNullable(rs.getBigDecimal("money")).orElse(BigDecimal.ZERO), params).get();
    }
}
