package ru.yandex.chemodan.app.psbilling.core.dao.groups.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.TrialDefinitionDao;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriod;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriodUnit;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.TrialDefinitionEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.TrialType;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;
import ru.yandex.misc.time.TimeUtils;

public class TrialDefinitionDaoImpl extends AbstractDaoImpl<TrialDefinitionEntity> implements TrialDefinitionDao {

    public TrialDefinitionDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "trial_definitions";
    }

    @Override
    public TrialDefinitionEntity parseRow(ResultSet rs) throws SQLException {
        return new TrialDefinitionEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                TrialType.R.fromValue(rs.getString("type")),
                rs.getBoolean("hidden"),
                rs.getBigDecimal("price"),
                Option.ofNullable(TimeUtils.getInstant(rs, "end_date")),
                CustomPeriod.fromResultSet(rs, "duration_measurement", "duration"),
                Option.ofNullable(rs.getString("single_usage_comparison_key"))
        );
    }

    @Override
    public ListF<TrialDefinitionEntity> findByIds(ListF<UUID> ids) {
        if (ids.isEmpty()) {
            return Cf.list();
        }
        return jdbcTemplate.query("select * from trial_definitions where id in ( :ids )", (rs, num) -> parseRow(rs),
                Cf.map("ids", ids));
    }

    @Override
    public TrialDefinitionEntity insert(InsertData insertData) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", Instant.now());
        params.put("type", insertData.getType().value());
        params.put("end_date", insertData.getEndDate().orElse((Instant) null));
        params.put("duration", insertData.getDuration().orElse((Integer) null));
        params.put("duration_measurement", insertData.getDurationMeasurement().map(CustomPeriodUnit::value)
                .orElse((String) null));
        params.put("price", insertData.getPrice());
        params.put("hidden", insertData.isHidden());
        params.put("singleUsageComparisonKey", insertData.getSingleUsageComparisonKey().orElse((String) null));

        return jdbcTemplate.queryForOption(
                "insert into trial_definitions " +
                        "(created_at,type,end_date,duration,duration_measurement,price, hidden, single_usage_comparison_key) " +
                        " values " +
                        "(:now, :type, :end_date, :duration, :duration_measurement,:price,:hidden, :singleUsageComparisonKey)" +
                        "RETURNING *",
                (rs, rowNum) -> parseRow(rs), params).get();
    }
}
