package ru.yandex.chemodan.app.psbilling.core.dao.groups.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.apache.commons.lang3.StringUtils;
import org.intellij.lang.annotations.Language;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.TrialUsageDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.TrialUsage;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.resultSet.ResultSetExtras;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class TrialUsageDaoImpl extends AbstractDaoImpl<TrialUsage> implements TrialUsageDao {
    public TrialUsageDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "trial_usage";
    }

    @Override
    public ListF<TrialUsage> findTrialUsages(Option<UUID> groupId, Option<PassportUid> uid,
                                             String trialUsageComparisonKey) {
        if (groupId.isEmpty() && uid.isEmpty()) {
            throw new IllegalArgumentException("both groupId and uid are empty");
        }

        MapF<String, Object> params = Cf.hashMap();
        params.put("comparison_key", trialUsageComparisonKey);

        String condition = "";
        if (groupId.isPresent()) {
            params.put("group_id", groupId);
            condition = "group_id = :group_id";
        }
        if (uid.isPresent()) {
            params.put("uid", uid.get().toString());
            if (!condition.isEmpty()) {
                condition += " or";
            }
            condition += " activated_by_uid = :uid";
        }

        return jdbcTemplate.query("select u.* from trial_usage u " +
                        " join trial_definitions td on u.trial_definition_id = td.id " +
                        " where td.single_usage_comparison_key = :comparison_key " +
                        "     and (" + condition + ")",
                (rs, num) -> parseRow(rs), params);
    }

    @Override
    public ListF<TrialUsage> findByTrialDefinitionAndGroup(UUID trialDefinitionId, UUID groupId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("trial_definition_id", trialDefinitionId);
        params.put("group_id", groupId);

        @Language("SQL")
        String sql = "select * from trial_usage " +
                " where trial_definition_id = :trial_definition_id " +
                " and group_id = :group_id ";

        return jdbcTemplate.query(sql, (rs, num) -> parseRow(rs), params);
    }

    @Override
    public TrialUsage parseRow(ResultSet rs) throws SQLException {
        String groupId = rs.getString("group_id");
        return new TrialUsage(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                StringUtils.isBlank(groupId) ? Option.empty() : Option.of(UUID.fromString(groupId)),
                new ResultSetExtras(rs).getLongO("activated_by_uid").map(PassportUid::cons),
                UUID.fromString(rs.getString("trial_definition_id")),
                new Instant(rs.getTimestamp("start_date")),
                new Instant(rs.getTimestamp("end_date"))
        );
    }

    @Override
    public TrialUsage insert(InsertData insertData) {
        if (insertData.getActivatedByUid().isEmpty() && insertData.getGroupId().isEmpty()) {
            throw new IllegalArgumentException("both group id and uid are empty");
        }

        MapF<String, Object> params = Cf.hashMap();
        params.put("group_id", insertData.getGroupId().orElse((UUID) null));
        params.put("trial_definition_id", insertData.getTrialDefinitionId());
        params.put("activated_by_uid", insertData.getActivatedByUid().orElse((PassportUid) null));
        params.put("now", Instant.now());
        params.put("end_date", insertData.getEndDate());

        @Language("SQL")
        String sql = "insert into trial_usage " +
                "(group_id,trial_definition_id,activated_by_uid,created_at,start_date,end_date ) " +
                " values " +
                "(:group_id, :trial_definition_id, :activated_by_uid, :now, :now, :end_date)" +
                "RETURNING *";

        return jdbcTemplate.queryForOption(sql, (rs, rowNum) -> parseRow(rs), params).get();
    }
}
