package ru.yandex.chemodan.app.psbilling.core.dao.groups.impl.billing;

import java.math.BigDecimal;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Currency;
import java.util.UUID;

import org.intellij.lang.annotations.Language;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.billing.ClientBalanceDao;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardPurpose;
import ru.yandex.chemodan.app.psbilling.core.entities.cards.CardStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupPaymentType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.ClientBalanceEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.PaymentInitiationType;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.balanceclient.model.response.CheckRequestPaymentResponse;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class ClientBalanceDaoImpl extends AbstractDaoImpl<ClientBalanceEntity> implements ClientBalanceDao {
    private static final Logger logger = LoggerFactory.getLogger(ClientBalanceDaoImpl.class);

    public ClientBalanceDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public ClientBalanceEntity insert(InsertData dataToInsert) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("clientId", dataToInsert.getClientId());
        params.put("balanceAmount", dataToInsert.getBalanceAmount());
        params.put("balanceCurrency", dataToInsert.getBalanceCurrency().getCurrencyCode());
        params.put("balanceVoidAt", dataToInsert.getBalanceVoidAt().orElse((Instant) null));
        params.put("lastInvoiceAt", dataToInsert.getLastInvoiceAt().orElse((Instant) null));
        params.put("now", Instant.now());

        return jdbcTemplate.query("insert into " + getTableName() + " " +
                "(client_id, balance_amount, balance_currency, balance_void_at, created_at, balance_updated_at, " +
                "last_invoice_at ) " +
                "values (:clientId, :balanceAmount, :balanceCurrency, :balanceVoidAt, :now, :now, :lastInvoiceAt) " +
                "returning *", (rs, rowNum) -> parseRow(rs), params).first();
    }

    @Override
    public Option<ClientBalanceEntity> find(Long clientId, Currency currency) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("clientId", clientId);
        params.put("currency", currency.getCurrencyCode());


        return jdbcTemplate.query("select * from " + getTableName() +
                        " where client_id=:clientId and balance_currency = :currency",
                (rs, rowNum) -> parseRow(rs), params).firstO();
    }

    @Override
    public Option<ClientBalanceEntity> findByGroupService(GroupService groupService) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("groupServiceId", groupService.getId());

        @Language("SQL")
        String query = "select cb.* " +
                " from group_services gs" +
                " join groups g on g.id = gs.group_id" +
                " join group_products gp on gp.id = gs.group_product_id" +
                " join client_balance cb on cb.client_id::text = g.payment_info->>'client_id'" +
                " where gs.id=:groupServiceId and balance_currency = gp.currency";
        return jdbcTemplate.query(
                query,
                (rs, rowNum) -> parseRow(rs), params).firstO();
    }

    @Override
    public ListF<ClientBalanceEntity> findByGroup(Group group) {
        if (group.getPaymentInfo().isEmpty()) {
            return Cf.list();
        }

        return jdbcTemplate.query(
                "select * from " + getTableName() + " where client_id=:clientId",
                (rs, rowNum) -> parseRow(rs),
                Cf.map("clientId", group.getPaymentInfo().get().getClientId())
        );
    }

    @Override
    public ClientBalanceEntity updateBalance(UUID id, BigDecimal balanceAmount) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("id", id);
        params.put("balanceAmount", balanceAmount);
        params.put("now", Instant.now());

        return jdbcTemplate.query("update " + getTableName() +
                " set balance_amount = :balanceAmount," +
                " balance_updated_at = :now" +
                " where id=:id" +
                " returning *", (rs, rowNum) -> parseRow(rs), params).first();
    }

    @Override
    public ClientBalanceEntity updateVoidAt(UUID id, Option<Instant> balanceVoidAt) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("id", id);
        params.put("balanceVoidAt", balanceVoidAt.orElse((Instant) null));
        params.put("now", Instant.now());

        return jdbcTemplate.query("update " + getTableName() +
                " set balance_void_at = :balanceVoidAt" +
                " where id=:id" +
                " returning *", (rs, rowNum) -> parseRow(rs), params).first();
    }

    @Override
    public ClientBalanceEntity updateInvoiceDate(UUID id, Option<Instant> lastInvoiceAt) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("id", id);
        params.put("lastInvoiceAt", lastInvoiceAt.orElse((Instant) null));
        params.put("now", Instant.now());

        return jdbcTemplate.query("update " + getTableName() +
                " set last_invoice_at = :lastInvoiceAt" +
                " where id=:id" +
                " returning *", (rs, rowNum) -> parseRow(rs), params).first();

    }

    @Override
    public ClientBalanceEntity createOrUpdate(Long clientId, Currency currency, BigDecimal balanceAmount,
                                              Option<Instant> balanceVoidAt) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("clientId", clientId);
        params.put("balanceCurrency", currency.getCurrencyCode());
        params.put("balanceAmount", balanceAmount);
        params.put("balanceVoidAt", balanceVoidAt.orElse((Instant) null));
        params.put("now", Instant.now());

        @Language("SQL") String sql = "" +
                "with old_value as (\n" +
                "    select *\n" +
                "    from client_balance\n" +
                "    where client_id = :clientId\n" +
                "      and balance_currency = :balanceCurrency\n" +
                "),\n" +
                "new_value as(\n" +
                "    insert into client_balance\n" +
                "    (client_id, balance_amount, balance_currency, balance_void_at, created_at, balance_updated_at)\n" +
                "    values (:clientId, :balanceAmount, :balanceCurrency, :balanceVoidAt, :now, :now)\n" +
                "        ON CONFLICT (client_id, balance_currency) DO UPDATE SET " +
                "            balance_amount = :balanceAmount,\n" +
                "            balance_void_at = :balanceVoidAt,\n" +
                "            balance_updated_at = :now\n" +
                "    returning *\n" +
                ")\n" +
                " select * from new_value" +
                " union all " +
                " select * from old_value";
        ListF<ClientBalanceEntity> result = jdbcTemplate.query(sql, (rs, rowNum) -> parseRow(rs), params);
        ClientBalanceEntity newValue = result.get(0);
        ClientBalanceEntity oldValue = result.length() > 1 ? result.get(1) : null;

        logger.info("updated client balance. old value:\n{};\nnew_value:\n{}", oldValue, newValue);
        return newValue;
    }

    @Override
    public ListF<Long> findObsoleteBalanceClients(Instant oldBalanceDate, Instant oldInvoiceDate,
                                                  Option<Long> greaterThan, int batchSize) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("batch_size", batchSize);
        params.put("balance_updated_at_threshold", oldBalanceDate);
        params.put("last_invoiced_at_threshold", oldInvoiceDate);

        String condition = "";
        if (greaterThan.isPresent()) {
            condition = " and client_id > :client_id\n";
            params.put("client_id", greaterThan.get());
        }

        @Language("SQL") String sql = "" +
                "select distinct client_id\n" +
                "from client_balance cb\n" +
                "where (cb.balance_updated_at < :balance_updated_at_threshold\n" +
                "   or cb.last_invoice_at > :last_invoiced_at_threshold)\n" +
                condition +
                "order by client_id\n" +
                "limit :batch_size";
        return jdbcTemplate.query(sql, (rs, rowNum) -> rs.getLong("client_id"), params);
    }

    @Override
    public int prepaidDebtsCount() {
        MapF<String, Object> params = Cf.hashMap();
        params.put("payment_type", GroupPaymentType.PREPAID);

        @Language("SQL") String sql = "" +
                "select count(1) as cnt\n" +
                "from groups g\n" +
                "         join group_services gs on g.id = gs.group_id\n" +
                "         join group_products gp on gs.group_product_id = gp.id\n" +
                "         join client_balance cb on gp.currency = cb.balance_currency" +
                "             and g.payment_info ->> 'client_id' = cb.client_id::text\n" +
                "where cb.balance_amount < 0\n" +
                "  and gp.payment_type = :payment_type::payment_type";
        return jdbcTemplate.query(sql, (rs, rowNum) -> rs.getInt("cnt"), params).first();
    }

    @Override
    public int countNotActualUpdatedBefore(Duration duration) {
        return jdbcTemplate.queryForObject(
                "SELECT count(*) FROM client_balance WHERE " +
                        "balance_updated_at < ?",
                Integer.class,
                Instant.now().minus(duration));
    }

    @Override
    public ListF<Tuple2<Long, Currency>> findClientsForAutoPay(Instant balanceVoidBefore, int batchSize,
                                                               Option<Tuple2<Long, Currency>> greaterThan) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("balance_void_before", balanceVoidBefore);
        params.put("limit", batchSize);
        params.put("target_enabled", Target.ENABLED);
        params.put("purpose", CardPurpose.B2B_PRIMARY);
        params.put("prepaid_type", GroupPaymentType.PREPAID);

        @Language("SQL") String condition = clientBatchFilterCondition(greaterThan, params);

        @Language("SQL")
        String sql =
                "select distinct cb.client_id, cb.balance_currency\n" +
                        "from client_balance cb\n" +
                        "         join groups g on g.payment_info ->> 'client_id' = cb.client_id :: text\n" +
                        "         join group_services gs on g.id = gs.group_id\n" +
                        "         join group_products gp on gp.id = gs.group_product_id\n" +
                        "where cb.balance_void_at <= :balance_void_before\n" +
                        "  and cb.balance_amount >= 0\n" +
                        "  and gs.skip_transactions_export = false\n" +
                        "  and gs.target = :target_enabled\n" +
                        // it's ok to find clients with pre and post paid services
                        // it is a mistake with will be handled by concrete auto payment task
                        "  and gp.payment_type = :prepaid_type :: payment_type\n" +
                        "  and g.payment_info->>'b2b_auto_billing_enabled' = 'true'\n" +
                        condition +
                        "group by cb.client_id, cb.balance_currency\n" +
                        "order by client_id, cb.balance_currency\n" +
                        "limit :limit;";

        return jdbcTemplate.query(sql, (rs, i) -> Tuple2.tuple(
                rs.getLong("client_id"),
                Currency.getInstance(rs.getString("balance_currency"))), params);
    }

    @Override
    public ListF<Tuple2<Long, Currency>> findClientsForAutoResurrectionPay(
            Instant balanceVoidAfter, Instant lastTryMinTime, double lastPaymentCoefficient,
            int batchSize, Option<Tuple2<Long, Currency>> greaterThan) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("balance_void_after", balanceVoidAfter);
        params.put("limit", batchSize);
        params.put("last_try_coefficient", lastPaymentCoefficient);
        params.put("last_try_min_time", lastTryMinTime);
        params.put("auto_payment_type", PaymentInitiationType.AUTO);
        params.put("no_funds_errors", CheckRequestPaymentResponse.NO_MONEY_ERRORS);
        params.put("target_disabled", Target.DISABLED);
        params.put("prepaid_type", GroupPaymentType.PREPAID);
        params.put("primary_purpose", CardPurpose.B2B_PRIMARY);
        params.put("active_card_status", CardStatus.ACTIVE);

        @Language("SQL") String batchFilterCondition = clientBatchFilterCondition(greaterThan, params);

        @Language("SQL")
        String sql = "select cb.client_id, cb.balance_currency\n" +
                "from client_balance cb\n" +
                "         join groups g on g.payment_info ->> 'client_id' = cb.client_id::text\n" +
                "         join cards c on c.uid = g.payment_info->>'uid'\n" +
                "         join group_services gs on g.id = gs.group_id\n" +
                "         join group_products gp on gs.group_product_id = gp.id\n" +
                "         left join group_trust_payment_requests gtpr on gtpr.client_id = cb.client_id\n" +
                "           and gtpr.currency = cb.balance_currency\n" +
                "           and gtpr.initiation_type = :auto_payment_type::payment_initiation_type\n" +
                "           and ((gtpr.error in (:no_funds_errors) and gtpr.payment_period_coefficient = :last_try_coefficient)\n" +
                "                or gtpr.error not in (:no_funds_errors)\n" +
                "               )\n" +
                "where cb.balance_amount <= 0\n" +
                "  and cb.balance_void_at >= :balance_void_after\n" +
                "  and g.payment_info ->> 'b2b_auto_billing_enabled' = 'true'\n" +
                "  and c.purpose = :primary_purpose::card_purpose\n" +
                "  and c.status = :active_card_status::card_status\n" +
                "  and gs.b_date = cb.balance_void_at\n" +
                "  and gs.target = :target_disabled\n" +
                "  and gs.skip_transactions_export = false\n" +
                "  and gp.currency = cb.balance_currency\n" +
                "  and gp.payment_type = :prepaid_type :: payment_type\n" +
                "  and gp.price > 0\n" +
                batchFilterCondition +
                "group by cb.client_id, cb.balance_currency\n" +
                "having max(gtpr.created_at) is null or max(gtpr.created_at) < :last_try_min_time\n" +
                "order by client_id, cb.balance_currency\n" +
                "limit :limit\n";

        return jdbcTemplate.query(sql, (rs, i) -> Tuple2.tuple(
                rs.getLong("client_id"),
                Currency.getInstance(rs.getString("balance_currency"))), params);
    }

    @NotNull
    private String clientBatchFilterCondition(Option<Tuple2<Long, Currency>> greaterThan, MapF<String, Object> params) {
        @Language("SQL")
        String condition = "";
        if (greaterThan.isPresent()) {
            params.put("client_id_from", greaterThan.get()._1);
            params.put("currency_from", greaterThan.get()._2.getCurrencyCode());
            condition = " and (cb.client_id > :client_id_from\n";
            condition += "    or cb.client_id = :client_id_from and cb.balance_currency > :currency_from)\n";
        }
        return condition;
    }

    @Override
    public String getTableName() {
        return "client_balance";
    }

    @Override
    public ClientBalanceEntity parseRow(ResultSet rs) throws SQLException {
        return new ClientBalanceEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                new Instant(rs.getTimestamp("balance_updated_at")),
                rs.getLong("client_id"),
                rs.getBigDecimal("balance_amount"),
                rs.getString("balance_currency"),
                Option.ofNullable(rs.getTimestamp("balance_void_at")).map(Instant::new),
                Option.ofNullable(rs.getTimestamp("last_invoice_at")).map(Instant::new));
    }
}
