package ru.yandex.chemodan.app.psbilling.core.dao.mail.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;

import lombok.AllArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.mail.EmailTemplateDao;
import ru.yandex.chemodan.app.psbilling.core.entities.mail.EmailTemplateEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.mail.LocalizedEmailTemplateEntity;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

@AllArgsConstructor
public class EmailTemplateDaoImpl implements EmailTemplateDao {
    protected final JdbcTemplate3 jdbcTemplate;
    private static final String EMAIL_TEMPLATES_TABLE = "email_templates";
    private static final String EMAIL_TEMPLATES_LOCALIZED_TABLE = "email_templates_localized";
    private static final String ARGS_DELIMITER = ";";

    public EmailTemplateEntity parseRow(ResultSet rs) throws SQLException {
        String argsVal = rs.getString("args");
        ListF<String> args = StringUtils.isBlank(argsVal) ? Cf.list() : Cf.list(argsVal.split(ARGS_DELIMITER));

        return new EmailTemplateEntity(
                rs.getString("key"),
                args,
                rs.getString("description"),
                new Instant(rs.getTimestamp("created_at")));
    }

    @Override
    public ListF<EmailTemplateEntity> findAll() {
        return jdbcTemplate.query("select * from " + EMAIL_TEMPLATES_TABLE,
                (rs, rowNum) -> parseRow(rs));
    }

    @Override
    public EmailTemplateEntity create(InsertData insertData) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("key", insertData.getKey());
        params.put("args", insertData.getArgs().isPresent()
                ? String.join(ARGS_DELIMITER, insertData.getArgs().get())
                : null);
        params.put("description", insertData.getDescription());
        params.put("now", Instant.now());

        return jdbcTemplate.queryForOption(
                "insert into " + EMAIL_TEMPLATES_TABLE +
                        " (created_at, key, description, args) " +
                        " values (:now, :key, :description, :args) RETURNING *",
                (rs, rowNum) -> parseRow(rs), params).get();
    }

    @Override
    public void mergeLocalizations(Collection<LocalizedEmailTemplateEntity> localizations) {
        jdbcTemplate.batchUpdate(
                "insert into " + EMAIL_TEMPLATES_LOCALIZED_TABLE + " (template_key, locale, sender_code)" +
                        " values (?, ?, ?)" +
                        " on conflict (template_key, locale)" +
                        "    DO UPDATE" +
                        "    SET sender_code = excluded.sender_code",
                Cf.toList(localizations).map(t -> new Object[]{t.getEmailTemplateKey(), t.getLocale(),
                        t.getSenderCode()})
        );
    }

    @Override
    public ListF<LocalizedEmailTemplateEntity> findLocalizations(String templateKey) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("template_key", templateKey);

        return jdbcTemplate.query(
                "select tl.* " +
                        "from " + EMAIL_TEMPLATES_LOCALIZED_TABLE + " tl " +
                        "    join email_templates t on t.key = tl.template_key " +
                        "where t.key = :template_key",
                (rs, num) -> parseLocalizationTemplate(rs), params);
    }

    @Override
    public Option<LocalizedEmailTemplateEntity> findLocalizedEmailTemplate(String templateKey, Language language) {
        return jdbcTemplate.queryForOption("select * from " + EMAIL_TEMPLATES_LOCALIZED_TABLE + " where template_key = ? and " +
                        "locale = ?",
                (rs, rowNum) -> parseLocalizationTemplate(rs), templateKey, language);
    }

    @Override
    public Option<EmailTemplateEntity> findByKeyO(String id) {
        return jdbcTemplate.queryForOption("select * from " + EMAIL_TEMPLATES_TABLE + " where key = ?",
                (rs, rowNum) -> parseRow(rs), id);
    }

    @NotNull
    @Override
    public EmailTemplateEntity findByKey(String key) {
        return findByKeyO(key).getOrThrow("Unable to find email template with key {}", key);
    }

    private static LocalizedEmailTemplateEntity parseLocalizationTemplate(ResultSet rs) throws SQLException {
        return new LocalizedEmailTemplateEntity(
                rs.getString("template_key"),
                rs.getString("locale"),
                rs.getString("sender_code")
        );
    }
}
