package ru.yandex.chemodan.app.psbilling.core.dao.products.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import javax.annotation.Nonnull;

import org.joda.time.Instant;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.products.FeatureDao;
import ru.yandex.chemodan.app.psbilling.core.entities.products.FeatureEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.FeatureType;
import ru.yandex.chemodan.app.psbilling.core.util.RequestTemplate;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class FeatureDaoImpl extends AbstractDaoImpl<FeatureEntity> implements FeatureDao {
    public FeatureDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "features";
    }

    @Nonnull
    @Override
    public FeatureEntity insert(FeatureEntity feature) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("code", feature.getCode());
        params.put("type", feature.getType().value());
        params.put("description", feature.getDescription());
        params.put("system_tvm_id", feature.getSystemTvmId().getOrNull());
        params.put("activation_url_http_method", feature.getActivationRequestTemplate()
                .map(RequestTemplate::getHttpMethod)
                .map(HttpMethod::name)
                .getOrNull()
        );
        params.put("activation_url_template", feature.getActivationRequestTemplate()
                .map(RequestTemplate::getUrlTemplate)
                .getOrNull()
        );
        params.put("activation_body_template", feature.getActivationRequestTemplate()
                .map(RequestTemplate::getBodyTemplate)
                .filter(Option::isPresent)
                .map(Option::get)
                .getOrNull()
        );
        params.put("set_amount_url_http_method", feature.getSetAmountRequestTemplate()
                .map(RequestTemplate::getHttpMethod)
                .map(HttpMethod::name)
                .getOrNull()
        );
        params.put("set_amount_url_template", feature.getSetAmountRequestTemplate()
                .map(RequestTemplate::getUrlTemplate)
                .getOrNull()
        );
        params.put("set_amount_body_template", feature.getSetAmountRequestTemplate()
                .map(RequestTemplate::getBodyTemplate)
                .filter(Option::isPresent)
                .map(Option::get)
                .getOrNull()
        );
        params.put("deactivation_url_http_method", feature.getDeactivationRequestTemplate()
                .map(RequestTemplate::getHttpMethod)
                .map(HttpMethod::name)
                .getOrNull()
        );
        params.put("deactivation_url_template", feature.getDeactivationRequestTemplate()
                .map(RequestTemplate::getUrlTemplate)
                .getOrNull()
        );
        params.put("deactivation_body_template", feature.getDeactivationRequestTemplate()
                .map(RequestTemplate::getBodyTemplate)
                .filter(Option::isPresent)
                .map(Option::get)
                .getOrNull()
        );
        params.put("call_set_amount_on_activation", feature.isCallSetAmountOnActivation());
        params.put("call_set_amount_on_deactivation", feature.isCallSetAmountOnDeactivation());
        params.put("now", now);
        params.put("error_processor_name", feature.getErrorProcessorName());

        UUID id = jdbcTemplate.query(
                "insert into features (created_at,updated_at,code,type,description,system_tvm_id,"
                        + "activation_url_http_method,activation_url_template,activation_body_template,"
                        + "set_amount_url_http_method,set_amount_url_template,set_amount_body_template,"
                        + "deactivation_url_http_method,deactivation_url_template,deactivation_body_template,"
                        + "call_set_amount_on_activation,call_set_amount_on_deactivation,error_processor_name) " +
                        "values(:now, :now,:code,:type,:description, :system_tvm_id, "
                        + ":activation_url_http_method,:activation_url_template,:activation_body_template,"
                        + ":set_amount_url_http_method,:set_amount_url_template,:set_amount_body_template,"
                        + ":deactivation_url_http_method,:deactivation_url_template,:deactivation_body_template,"
                        + ":call_set_amount_on_activation,:call_set_amount_on_deactivation,:error_processor_name) " +
                        "RETURNING id",
                (rs, num) -> UUID.fromString(rs.getString("id")), params).first();
        return feature.toBuilder().createdAt(now).updatedAt(now).id(id).build();
    }

    @Override
    public FeatureEntity parseRow(ResultSet rs) throws SQLException {
        return new FeatureEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                new Instant(rs.getTimestamp("updated_at")),
                rs.getString("code"),
                Option.ofNullable(rs.getInt("system_tvm_id")),
                FeatureType.R.fromValue(rs.getString("type")),
                rs.getString("description"),
                parseRequestTemplate(rs, "activation"),
                parseRequestTemplate(rs, "set_amount"),
                parseRequestTemplate(rs, "deactivation"),
                rs.getBoolean("call_set_amount_on_activation"),
                rs.getBoolean("call_set_amount_on_deactivation"),
                rs.getString("error_processor_name")
        );
    }

    private static Option<RequestTemplate> parseRequestTemplate(ResultSet rs, String prefix) throws SQLException {
        String template = rs.getString(prefix + "_url_template");
        if (template != null) {
            return Option.of(new RequestTemplate(
                    HttpMethod.valueOf(rs.getString(prefix + "_url_http_method")),
                    template,
                    Option.ofNullable(rs.getString(prefix + "_body_template")),
                    Option.ofNullable(rs.getString(prefix + "_content_type")).map(MediaType::valueOf)
            ));
        } else {
            return Option.empty();
        }
    }

    @Override
    public boolean isEnabledForGroup(UUID organizationId, String featureCode) {

        return jdbcTemplate.query(
                "select exists" +
                        "(select f.code from group_service_features gsf\n" +
                        "join product_features pf on gsf.product_feature_id = pf.id\n" +
                        "join features f on pf.feature_id = f.id\n" +
                        "where f.code = ?\n" +
                        "and pf.enabled = 'true'" +
                        "and gsf.group_id = ?\n" +
                        "and gsf.target = 'enabled');",
                (rs, rowNum) -> rs.getBoolean(1),
                featureCode,
                organizationId
        ).first();
    }
}
