package ru.yandex.chemodan.app.psbilling.core.dao.products.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import javax.annotation.Nonnull;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.entities.products.FeatureScope;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductFeatureEntity;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class ProductFeatureDaoImpl extends AbstractDaoImpl<ProductFeatureEntity> implements ProductFeatureDao {
    public ProductFeatureDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "product_features";
    }

    @Override
    public MapF<UUID, ListF<ProductFeatureEntity>> findEnabledByUserProductIds(ListF<UUID> userProductIds) {
        if (userProductIds.isEmpty()) {
            return Cf.map();
        }
        return jdbcTemplate
                .query("select * from product_features where user_product_id in ( :ids ) and enabled = true order by " +
                                "order_num",
                        (rs, num) -> parseRow(rs), Cf.map("ids", userProductIds))
                .groupBy(ProductFeatureEntity::getUserProductId);
    }

    @Override
    public ListF<ProductFeatureEntity> findByIds(CollectionF<UUID> ids) {
        if (ids.isEmpty()) {
            return Cf.list();
        }
        return jdbcTemplate.query(
                "select * from product_features where id in (:ids)",
                (rs, i) -> parseRow(rs),
                Cf.map("ids", ids)
        );
    }

    @Override
    public void disable(UUID productFeatureId) {
        jdbcTemplate.update(
                "update product_features set enabled = false where id =:id",
                Cf.map("id", productFeatureId)
        );
    }

    @Nonnull
    @Override
    public ProductFeatureEntity insert(InsertData dataToInsert) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("user_product_id", dataToInsert.getUserProductId());
        params.put("feature_id", dataToInsert.getFeatureId() == null ? null : dataToInsert.getFeatureId().getOrNull());
        params.put("amount", dataToInsert.getAmount());
        params.put("description_key_id", dataToInsert.getDescriptionTankerKeyId() == null ?
                null : dataToInsert.getDescriptionTankerKeyId().getOrNull());
        params.put("value_key_id", dataToInsert.getValueTankerKeyId() == null ?
                null : dataToInsert.getValueTankerKeyId().getOrNull());
        params.put("order_num", dataToInsert.getOrderNum());
        params.put("now", now);
        params.put("code", dataToInsert.getCode());
        params.put("scope", dataToInsert.getScope().value());

        return jdbcTemplate.query(
                "insert into product_features (created_at,"
                        + "user_product_id,feature_id,amount, description_tanker_key_id, value_tanker_key_id, " +
                        "order_num, code, " +
                        "scope) " +
                        "values(:now,:user_product_id,:feature_id,:amount, :description_key_id, :value_key_id," +
                        " :order_num, :code, " +
                        ":scope::feature_scope) " +
                        "RETURNING *",
                (rs, num) -> parseRow(rs), params).first();
    }

    @Override
    public ProductFeatureEntity parseRow(ResultSet rs) throws SQLException {
        String descriptionTankerKeyId = rs.getString("description_tanker_key_id");
        String groupTankerKeyId = rs.getString("group_tanker_key_id");
        String valueTankerKeyId = rs.getString("value_tanker_key_id");
        String featureId = rs.getString("feature_id");
        return new ProductFeatureEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                StringUtils.isBlank(featureId) ? Option.empty() : Option.of(UUID.fromString(featureId)),
                UUID.fromString(rs.getString("user_product_id")),
                rs.getBigDecimal("amount"),
                rs.getBoolean("enabled"),
                rs.getInt("order_num"),
                StringUtils.isNotEmpty(descriptionTankerKeyId) ?
                        Option.of(UUID.fromString(descriptionTankerKeyId)) : Option.empty(),
                StringUtils.isNotEmpty(groupTankerKeyId) ?
                        Option.of(UUID.fromString(groupTankerKeyId)) : Option.empty(),
                StringUtils.isNotEmpty(valueTankerKeyId) ?
                        Option.of(UUID.fromString(valueTankerKeyId)) : Option.empty(),
                rs.getString("code"),
                FeatureScope.R.fromValue(rs.getString("scope"))
        );
    }
}
