package ru.yandex.chemodan.app.psbilling.core.dao.products.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Instant;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductLineDao;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductLineEntity;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class ProductLineDaoImpl extends AbstractDaoImpl<ProductLineEntity> implements ProductLineDao {

    public ProductLineDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "product_lines";
    }

    @Override
    public ProductLineEntity parseRow(ResultSet rs) throws SQLException {
        return new ProductLineEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                UUID.fromString(rs.getString("product_set_id")),
                rs.getString("selector_bean_el"),
                rs.getInt("order_num"),
                rs.getString("description")
        );
    }

    @Override
    @Transactional
    public void bindUserProducts(UUID productLineId, ListF<UUID> userProductIds) {
        jdbcTemplate.update("delete from user_products_to_product_lines where product_line_id = ?", productLineId);
        jdbcTemplate.batchUpdate(
                "insert into user_products_to_product_lines(product_line_id, user_product_id, order_num)" +
                        " values (?, ?, ?)",
                userProductIds.zipWithIndex().map((productId, index) -> new Object[]{productLineId, productId, index})
        );
    }

    @Override
    @Transactional
    public void bindGroupProducts(UUID productLineId, ListF<UUID> groupProductIds) {
        jdbcTemplate.update("delete from group_products_to_product_lines where product_line_id = ?", productLineId);
        jdbcTemplate.batchUpdate(
                "insert into group_products_to_product_lines(product_line_id, group_product_id, order_num)" +
                        " values (?, ?, ?)",
                groupProductIds.zipWithIndex().map((productId, index) -> new Object[]{productLineId, productId, index})
        );
    }

    @Override
    public ListF<ProductLineEntity> findByProductSetId(UUID productSetId) {
        return jdbcTemplate.query(
                "select * from product_lines where product_set_id = ? order by order_num",
                (rs, num) -> parseRow(rs), productSetId
        );
    }

    @Override
    public ListF<ProductLineEntity> findByProductSetIds(CollectionF<UUID> productSetIds) {
        if (productSetIds.isEmpty()) {
            return Cf.list();
        }
        return jdbcTemplate.query(
                "select * from product_lines where product_set_id in (:ids) order by product_set_id, order_num",
                (rs, num) -> parseRow(rs), Cf.map("ids", productSetIds)
        );
    }

    @Override
    public ListF<ProductLineEntity> findByUserProduct(UUID userProductId) {
        return jdbcTemplate.query("select p.* from " +
                " product_lines p join user_products_to_product_lines p_l on p_l.product_line_id = p.id " +
                " where p_l.user_product_id  = :id" +
                " order by p_l.order_num ",
                (rs, num) -> parseRow(rs),
                Cf.map("id", userProductId));
    }

    @Override
    public ListF<ProductLineEntity> findByGroupProduct(UUID groupProductId) {
        return jdbcTemplate.query("select p.* from " +
                        " product_lines p join group_products_to_product_lines p_l on p_l.product_line_id = p.id " +
                        " where p_l.group_product_id  = :id" +
                        " order by p_l.order_num ",
                (rs, num) -> parseRow(rs),
                Cf.map("id", groupProductId));
    }

    @Override
    public ProductLineEntity create(InsertData dataToInsert) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("product_set_id", dataToInsert.getProductSetId());
        params.put("num", dataToInsert.getOrderNum());
        params.put("selector_bean_el",dataToInsert.getSelectorBeanEL());
        params.put("now", now);
        params.put("description", dataToInsert.getDescription().getOrNull());

        return jdbcTemplate.queryForOption("insert into product_lines" +
                        " (product_set_id, selector_bean_el, order_num, created_at, description) " +
                        " values (:product_set_id,:selector_bean_el ,:num, :now, :description) RETURNING *",
                (rs, rowNum) -> parseRow(rs), params).get();
    }

}
