package ru.yandex.chemodan.app.psbilling.core.dao.products.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductOwnerDao;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductOwner;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class ProductOwnerDaoImpl extends AbstractDaoImpl<ProductOwner> implements ProductOwnerDao {
    public ProductOwnerDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "product_owners";
    }

    @Override
    public ProductOwner parseRow(ResultSet rs) throws SQLException {
        String defaultGroupProductId = rs.getString("default_group_product_id");
        return new ProductOwner(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                rs.getString("code"),
                StringUtils.isBlank(defaultGroupProductId) ? Option.empty() :
                        Option.of(UUID.fromString(defaultGroupProductId))
        );
    }

    @Override
    public ListF<ProductOwner> findByIds(ListF<UUID> ids) {
        if (ids.isEmpty()) {
            return Cf.list();
        }
        return jdbcTemplate.query("select * from product_owners where id in ( :ids )",
                (rs, num) -> parseRow(rs), Cf.map("ids", ids));
    }

    @Override
    public Option<ProductOwner> findByCode(String code) {
        return jdbcTemplate.queryForOption("select * from product_owners where code = :code",
                (rs, num) -> parseRow(rs), Cf.map("code", code));
    }

    @Override
    public ProductOwner create(String code, Option<UUID> defaultGroupProductId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("code", code);
        params.put("now", Instant.now());
        params.put("defaultGroupProductId", defaultGroupProductId.getOrNull());

        return jdbcTemplate.queryForOption("insert into product_owners" +
                        " (code, created_at, default_group_product_id) " +
                        " values (:code, :now, :defaultGroupProductId) RETURNING *",
                (rs, rowNum) -> parseRow(rs), params).get();
    }
}
