package ru.yandex.chemodan.app.psbilling.core.dao.products.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductTemplateFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductTemplateFeatureEntity;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

/**
 * @author friendlyevil
 */
public class ProductTemplateFeatureDaoImpl extends AbstractDaoImpl<ProductTemplateFeatureEntity> implements ProductTemplateFeatureDao {
    public ProductTemplateFeatureDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "product_template_features";
    }

    @Override
    public ProductTemplateFeatureEntity insert(InsertData insertData) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("product_template_id", insertData.getProductTemplateId());
        params.put("feature_id", insertData.getFeatureId());
        params.put("amount", insertData.getAmount());
        params.put("now", now);
        params.put("description_tanker_key_id", insertData.getDescriptionTankerKeyId().orElse((UUID) null));
        params.put("order_num", insertData.getOrderNum());
        params.put("code", insertData.getCode());

        return jdbcTemplate.query("insert into product_template_features (product_template_id, feature_id, amount, " +
                        "created_at, description_tanker_key_id, order_num, code) " +
                        "values (:product_template_id, :feature_id, :amount, :now, :description_tanker_key_id, " +
                        ":order_num, :code) " +
                        "RETURNING *",
                (rs, num) -> parseRow(rs), params).first();
    }

    @Override
    public Option<ProductTemplateFeatureEntity> findByProductFeatureId(UUID productFeatureId) {
        return jdbcTemplate.queryForOption("select ptf.* from product_features as pf " +
                        "join user_products as up on pf.user_product_id = up.id " +
                        "join product_templates as pt on pt.code = up.code_family " +
                        "join product_template_features as ptf on " +
                        "ptf.product_template_id = pt.id and ptf.feature_id = pf.feature_id " +
                        "where pf.id = :productFeatureId",
                (rs, num) -> parseRow(rs), Cf.map("productFeatureId", productFeatureId)
        );
    }

    @Override
    public ProductTemplateFeatureEntity parseRow(ResultSet rs) throws SQLException {
        String descriptionTankerKeyId = rs.getString("description_tanker_key_id");
        String groupTankerKeyId = rs.getString("group_tanker_key_id");
        String valueTankerKeyId = rs.getString("value_tanker_key_id");
        String featureId = rs.getString("feature_id");
        return new ProductTemplateFeatureEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                UUID.fromString(rs.getString("product_template_id")),
                StringUtils.isBlank(featureId) ? Option.empty() : Option.of(UUID.fromString(featureId)),
                rs.getBigDecimal("amount"),
                rs.getBoolean("enabled"),
                rs.getInt("order_num"),
                StringUtils.isNotEmpty(descriptionTankerKeyId) ?
                        Option.of(UUID.fromString(descriptionTankerKeyId)) : Option.empty(),
                StringUtils.isNotEmpty(groupTankerKeyId) ?
                        Option.of(UUID.fromString(groupTankerKeyId)) : Option.empty(),
                StringUtils.isNotEmpty(valueTankerKeyId) ?
                        Option.of(UUID.fromString(valueTankerKeyId)) : Option.empty(),
                rs.getString("code")
        );
    }
}
