package ru.yandex.chemodan.app.psbilling.core.dao.products.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import javax.annotation.Nonnull;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductDao;
import ru.yandex.chemodan.app.psbilling.core.entities.products.BillingType;
import ru.yandex.chemodan.app.psbilling.core.entities.products.UserProductEntity;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class UserProductDaoImpl extends AbstractDaoImpl<UserProductEntity> implements UserProductDao {

    public UserProductDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    public Option<UserProductEntity> findByCodeO(String code) {
        return jdbcTemplate.queryForOption("select * from " + getTableName() + " where code = ?",
                (rs, rowNum) -> parseRow(rs), code);
    }

    public UserProductEntity findByCode(String code) {
        return jdbcTemplate.queryForObject("select * from " + getTableName() + " where code = ?",
                (rs, rowNum) -> parseRow(rs), code);
    }

    @Nonnull
    @Override
    public UserProductEntity insert(InsertData dataToInsert) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("code", dataToInsert.getCode());
        params.put("code_family", dataToInsert.getCodeFamily());
        params.put("titleKeyId", dataToInsert.getTitleTankerKeyId());
        params.put("billing_type", dataToInsert.getBillingType().value());
        params.put("allow_auto_prolong", dataToInsert.isAllowAutoProlong());
        params.put("now", now);
        params.put("best_offer", dataToInsert.isBestOffer());
        params.put("owner_id", dataToInsert.getProductOwnerId());
        params.put("available_from",
                dataToInsert.getAvailableForm() == null ? null : dataToInsert.getAvailableForm().getOrNull());
        params.put("trust_subs_retry_charging_limit", dataToInsert.getTrustSubsChargingRetryLimit() == null ? null :
                dataToInsert.getTrustSubsChargingRetryLimit().getOrNull());
        params.put("trust_subs_retry_charging_delay", dataToInsert.getTrustSubsChargingRetryDelay() == null ? null :
                dataToInsert.getTrustSubsChargingRetryDelay().getOrNull());
        params.put("trust_subs_grace_period", dataToInsert.getTrustSubsGracePeriod() == null ? null :
                dataToInsert.getTrustSubsGracePeriod().getOrNull());
        params.put("singleton", dataToInsert.isSingleton());
        params.put("trust_service_id", dataToInsert.getTrustServiceId() == null ? null :
                dataToInsert.getTrustServiceId().getOrNull());
        params.put("trial_definition_id",
                dataToInsert.getTrialDefinitionId() == null ? null : dataToInsert.getTrialDefinitionId().getOrNull());


        return jdbcTemplate.queryForOption(
                "insert into user_products (code, code_family, title_tanker_key_id, billing_type, is_best_offer, " +
                        "product_owner_id," +
                        "created_at,updated_at, available_from, allow_auto_prolong, trust_subs_retry_charging_limit, " +
                        "trust_subs_retry_charging_delay,trust_subs_grace_period, singleton, trust_service_id, " +
                        "trial_definition_id) " +
                        "values(:code, :code_family, :titleKeyId, :billing_type, :best_offer, :owner_id, :now, :now, " +
                        ":available_from," +
                        " :allow_auto_prolong, :trust_subs_retry_charging_limit, :trust_subs_retry_charging_delay," +
                        " :trust_subs_grace_period, :singleton, :trust_service_id, :trial_definition_id ) " +
                        "RETURNING *",
                (rs, num) -> parseRow(rs), params).get();
    }

    @Override
    public ListF<UserProductEntity> findByIds(ListF<UUID> userProductIds) {
        if (userProductIds.isEmpty()) {
            return Cf.list();
        }
        return jdbcTemplate.query("select * from user_products where id in ( :ids )", (rs, num) -> parseRow(rs),
                Cf.map("ids", userProductIds));
    }

    @Override
    public UserProductEntity findByGroupServiceId(UUID groupServiceId) {
        return jdbcTemplate.queryForOption("select up.* from group_products gp " +
                        " join group_services gs on gp.id = gs.group_product_id " +
                        " join user_products up on gp.user_product_id = up.id " +
                        " where gs.id = ?",
                (rs, rowNum) -> parseRow(rs), groupServiceId).get();
    }

    @Override
    public ListF<UserProductEntity> findByBillingTypes(BillingType... billingTypes)
    {
        if (billingTypes == null || billingTypes.length == 0) {
            return jdbcTemplate.query("select * from user_products", (rs, num) -> parseRow(rs));
        } else {
            return jdbcTemplate
                    .query("select * from user_products where billing_type in ( :types )", (rs, num) -> parseRow(rs),
                            Cf.map("types", Cf.x(billingTypes).map(BillingType::value)));
        }
    }

    @Override
    public ListF<UserProductEntity> findByProductLine(UUID lineId) {
        return jdbcTemplate.query("select u.* from " +
                        " user_products u join user_products_to_product_lines p_l on p_l.user_product_id = u.id " +
                        " where p_l.product_line_id  = :id" +
                        " order by p_l.order_num ",
                (rs, num) -> parseRow(rs)
                , Cf.map("id", lineId));
    }

    @Override
    public String getTableName() {
        return "user_products";
    }

    public UserProductEntity parseRow(ResultSet rs) throws SQLException {
        String tankerKeyId = rs.getString("title_tanker_key_id");
        String subsGracePeriod = rs.getString("trust_subs_grace_period");
        String retryChargingDelay = rs.getString("trust_subs_retry_charging_delay");
        String retryChargingLimit = rs.getString("trust_subs_retry_charging_limit");
        String trustServiceId = rs.getString("trust_service_id");
        String trialDefinitionId = rs.getString("trial_definition_id");

        return new UserProductEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                StringUtils.isBlank(tankerKeyId) ? Option.empty() : Option.of(UUID.fromString(tankerKeyId)),
                StringUtils.isBlank(trustServiceId) ? Option.empty() : Option.of(Integer.parseInt(trustServiceId)),
                rs.getString("code"),
                rs.getString("code_family"),
                BillingType.R.fromValue(rs.getString("billing_type")),
                rs.getBoolean("is_best_offer"),
                UUID.fromString(rs.getString("product_owner_id")),
                Option.ofNullable(rs.getTimestamp("available_from")).map(Instant::new),
                rs.getBoolean("allow_auto_prolong"),
                StringUtils.isBlank(subsGracePeriod) ? Option.empty() : Option.of(subsGracePeriod),
                StringUtils.isBlank(retryChargingDelay) ? Option.empty() : Option.of(retryChargingDelay),
                StringUtils.isBlank(retryChargingLimit) ? Option.empty() : Option.of(retryChargingLimit),
                StringUtils.isBlank(trialDefinitionId) ? Option.empty() : Option.of(UUID.fromString(trialDefinitionId))
        );
    }

}
