package ru.yandex.chemodan.app.psbilling.core.dao.products.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.apache.commons.lang3.StringUtils;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductPeriodDao;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriod;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriodUnit;
import ru.yandex.chemodan.app.psbilling.core.entities.products.UserProductPeriodEntity;
import ru.yandex.misc.db.resultSet.ResultSetExtras;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class UserProductPeriodDaoImpl extends AbstractDaoImpl<UserProductPeriodEntity> implements UserProductPeriodDao {
    public UserProductPeriodDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "user_product_periods";
    }

    @Override
    public UserProductPeriodEntity parseRow(ResultSet rs) throws SQLException {
        String trustFiscalTitle = rs.getString("trust_fiscal_title");
        String packageName = rs.getString("package_name");
        ResultSetExtras extras = new ResultSetExtras(rs);
        return new UserProductPeriodEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                rs.getString("code"),
                new CustomPeriod(CustomPeriodUnit.R.fromValue(rs.getString("period")), rs.getInt("period_count")),
                UUID.fromString(rs.getString("user_product_id")),
                StringUtils.isBlank(trustFiscalTitle) ? Option.empty() : Option.of(trustFiscalTitle),
                StringUtils.isBlank(packageName) ? Option.empty() : Option.of(packageName),
                CustomPeriod.fromResultSet(rs, "start_period_discount_duration_measurement",
                        "start_period_discount_duration_length"),
                extras.getIntO("start_period_count"));
    }

    @Override
    public UserProductPeriodEntity create(InsertData dataToInsert) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("code", dataToInsert.getCode());
        params.put("period", dataToInsert.getPeriod().getUnit().value());
        params.put("period_count", dataToInsert.getPeriod().getValue());
        params.put("trust_fiscal_title", dataToInsert.getTrustFiscalTitle().orElse((String) null));
        params.put("user_product_id", dataToInsert.getUserProductId());
        params.put("package_name", dataToInsert.getPackageName().orElse((String) null));
        params.put("now", now);
        params.put("start_period_discount_duration_measurement",
                dataToInsert.getStartPeriodDurationMeasurement().map(CustomPeriodUnit::value).orElse((String) null));
        params.put("start_period_discount_duration_length",
                dataToInsert.getStartPeriodDurationLength().orElse((Integer) null));
        params.put("start_period_count", dataToInsert.getStartPeriodCount().orElse((Integer) null));
        return jdbcTemplate.queryForOption(
                "insert into user_product_periods (created_at,user_product_id,period,code,trust_fiscal_title, " +
                        "package_name, start_period_discount_duration_measurement, " +
                        "start_period_discount_duration_length, " +
                        "start_period_count) " +
                        "values(:now, :user_product_id, :period, :code, :trust_fiscal_title, :package_name, " +
                        ":start_period_discount_duration_measurement::duration_measurement, " +
                        ":start_period_discount_duration_length, " +
                        ":start_period_count) " +
                        "RETURNING *",
                (rs, num) -> parseRow(rs), params).get();
    }

    @Override
    public MapF<UUID, ListF<UserProductPeriodEntity>> findByUserProductIds(ListF<UUID> userProductIds) {
        if (userProductIds.isEmpty()) {
            return Cf.map();
        }
        return jdbcTemplate.query("select * from user_product_periods where user_product_id in ( :ids )",
                        (rs, num) -> parseRow(rs), Cf.map("ids", userProductIds))
                .groupBy(UserProductPeriodEntity::getUserProductId);
    }

    @Override
    public Option<UserProductPeriodEntity> findByCode(String code) {
        return jdbcTemplate.queryForOption("select * from user_product_periods where code= :code",
                (rs, num) -> parseRow(rs), Cf.map("code", code));
    }

    @Override
    public ListF<UserProductPeriodEntity> findByIds(ListF<UUID> ids) {
        if (ids.isEmpty()) {
            return Cf.list();
        }
        return jdbcTemplate.query("select * from user_product_periods where id in ( :ids )",
                (rs, num) -> parseRow(rs), Cf.map("ids", ids));
    }
}
