package ru.yandex.chemodan.app.psbilling.core.dao.products.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductPricesDao;
import ru.yandex.chemodan.app.psbilling.core.entities.products.UserProductPriceEntity;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class UserProductPricesDaoImpl extends AbstractDaoImpl<UserProductPriceEntity> implements UserProductPricesDao {
    public UserProductPricesDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "user_product_prices";
    }

    @Override
    public UserProductPriceEntity parseRow(ResultSet rs) throws SQLException {
        return new UserProductPriceEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                UUID.fromString(rs.getString("user_product_period_id")),
                rs.getString("region_id"),
                rs.getBigDecimal("price"),
                rs.getString("currency"),
                Option.ofNullable(rs.getBigDecimal("display_discount_percent")),
                Option.ofNullable(rs.getBigDecimal("display_original_price")),
                Option.ofNullable(rs.getBigDecimal("start_period_price"))
        );
    }

    @Override
    public MapF<UUID, ListF<UserProductPriceEntity>> findByPeriodIds(ListF<UUID> userPeriodIds) {
        if (userPeriodIds.isEmpty()) {
            return Cf.map();
        }
        return jdbcTemplate.query("select * from user_product_prices where user_product_period_id in ( :ids )",
                (rs, num) -> parseRow(rs), Cf.map("ids", userPeriodIds))
                .groupBy(UserProductPriceEntity::getUserProductPeriodId);
    }

    @Override
    public ListF<UserProductPriceEntity> findPricesByPeriodCode(String userPeriodCode) {
        return jdbcTemplate.query("select * from user_product_prices where user_product_period_id in (" +
                                "select id from user_product_periods where code = :code" +
                                ")",
                        (rs, num) -> parseRow(rs), Cf.map("code", userPeriodCode));
    }

    @Override
    public ListF<UserProductPriceEntity> findByIds(ListF<UUID> ids) {
        if(ids.isEmpty()){
            return Cf.list();
        }
        return jdbcTemplate.query("select * from user_product_prices where id in ( :ids )",
                (rs, num) -> parseRow(rs), Cf.map("ids", ids));
    }

    @Override
    public UserProductPriceEntity create(InsertData dataToInsert) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("user_product_period_id", dataToInsert.getUserProductPeriodId());
        params.put("region_id", dataToInsert.getRegionId());
        params.put("price", dataToInsert.getPrice());
        params.put("currency", dataToInsert.getCurrencyCode());
        params.put("now", now);
        params.put("display_discount_percent", getNullableOptionValue(dataToInsert.getDisplayDiscountPercent()));
        params.put("display_original_price", getNullableOptionValue(dataToInsert.getDisplayOriginalPrice()));
        params.put("start_period_price", getNullableOptionValue(dataToInsert.getStartPeriodPrice()));

        return jdbcTemplate.queryForOption(
                "insert into user_product_prices (created_at,user_product_period_id,region_id,price,currency, " +
                        "display_discount_percent, display_original_price, start_period_price) " +
                        "values(:now, :user_product_period_id, :region_id, :price, :currency, " +
                        ":display_discount_percent, :display_original_price, :start_period_price) " +
                        "RETURNING *",
                (rs, num) -> parseRow(rs), params).get();
    }

    private <T> T getNullableOptionValue(Option<T> value) {
        return value == null ? null : value.getOrNull();
    }
}
