package ru.yandex.chemodan.app.psbilling.core.dao.promocodes.impl;

import java.sql.Array;
import java.sql.JDBCType;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.UUID;

import org.intellij.lang.annotations.Language;
import org.joda.time.Instant;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.dao.support.DataAccessUtils;
import org.springframework.jdbc.core.BatchPreparedStatementSetter;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractPKBasedDAOImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.promocodes.PromoCodeDao;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.PromoCodeEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.PromoCodeStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.PromoCodeType;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.SafePromoCode;
import ru.yandex.misc.db.resultSet.ResultSetExtras;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class PromoCodeDaoImpl extends AbstractPKBasedDAOImpl<PromoCodeEntity, SafePromoCode> implements PromoCodeDao {
    @Language("SQL")
    public static final String INSERT_QUERY = "" +
            "insert into promo_codes(" +
            " code, " +
            " user_product_price_id, " +
            " promo_template_id, " +
            " num_activations, " +
            "remaining_activations, " +
            " status, " +
            " status_updated_at, " +
            " status_reason, " +
            " from_date, " +
            " to_date, " +
            " created_at, " +
            " updated_at, " +
            " type, " +
            " promo_code_template_code " +
            ") values (" +
            " ?," +
            " ?," +
            " ?," +
            " ?," +
            " ?," +
            " ?::promo_code_status," +
            " ?," +
            " ?," +
            " ?," +
            " ?," +
            " ?," +
            " ?," +
            " ?::promo_codes_type," +
            " ? " +
            ") returning *";

    public PromoCodeDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    //method that doesn't allocate an InsertData object for each of 1e5 promo codes
    public void create(InsertData dataToInsert) {
        Timestamp now = new Timestamp(Instant.now().getMillis());
        jdbcTemplate.batchUpdate(INSERT_QUERY, new BatchPreparedStatementSetter() {
            @Override
            public void setValues(PreparedStatement ps, int i) throws SQLException {
                dataToInsert.getCodes().get(i).setValue(ps, 1);

                ps.setObject(2, dataToInsert.getProductPriceId().getOrNull());
                ps.setObject(3, dataToInsert.getPromoTemplateId().getOrNull());
                ps.setObject(4, dataToInsert.getNumActivations().getOrNull());
                ps.setObject(5, dataToInsert.getRemainingActivations().getOrNull());
                ps.setObject(6, dataToInsert.getPromoCodeStatus().value());
                ps.setTimestamp(7, new Timestamp(dataToInsert.getStatusUpdatedAt().getMillis()));
                ps.setObject(8, dataToInsert.getStatusReason().getOrNull());
                ps.setTimestamp(9, new Timestamp(dataToInsert.getFromDate().getMillis()));
                ps.setTimestamp(10, dataToInsert.getToDate().map(Instant::getMillis).map(Timestamp::new).getOrNull());
                ps.setTimestamp(11, now);
                ps.setTimestamp(12, now);
                ps.setObject(13, dataToInsert.getPromoCodeType().value());
                ps.setObject(14, dataToInsert.getTemplateCode());
            }

            @Override
            public int getBatchSize() {
                return dataToInsert.getCodes().size();
            }
        });
    }

    @Override
    public Option<PromoCodeEntity> findByIdAndTypeO(SafePromoCode promoCode, PromoCodeType type) {
        MapF<String, Object> params = Cf.map(
                "code", promoCode,
                "type", type
        );

        @Language("SQL")
        String sql = "SELECT * FROM promo_codes WHERE code = :code AND type = :type::promo_codes_type";

        return jdbcTemplate.queryForOption(sql, (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public PromoCodeEntity parseRow(ResultSet rs) throws SQLException {
        ResultSetExtras extras = new ResultSetExtras(rs);
        return new PromoCodeEntity(
                SafePromoCode.cons(rs.getString("code")),
                Option.ofNullable(rs.getObject("user_product_price_id", UUID.class)),
                Option.ofNullable(rs.getObject("promo_template_id", UUID.class)),
                extras.getIntO("num_activations"),
                extras.getIntO("remaining_activations"),
                PromoCodeStatus.R.fromValue(rs.getString("status")),
                new Instant(rs.getTimestamp("status_updated_at")),
                Option.ofNullable(rs.getString("status_reason")),
                new Instant(rs.getTimestamp("from_date")),
                Option.ofNullable(rs.getTimestamp("to_date")).map(Instant::new),
                new Instant(rs.getTimestamp("created_at")),
                new Instant(rs.getTimestamp("updated_at")),
                Option.ofNullable(rs.getString("type"))
                        .map(PromoCodeType.R::fromValue)
                        .orElse(PromoCodeType.B2C),
                Option.ofNullable(rs.getString("promo_code_template_code"))
        );
    }

    @Override
    public CollectionF<SafePromoCode> getConflictingCodes(CollectionF<SafePromoCode> toCheck) {
        String[] codes = toCheck.map(SafePromoCode::getOriginalPromoCode).toArray(String.class);

        return jdbcTemplate.query("select distinct tt.code\n" +
                        "from (select * from unnest(?::text[]) code) tt\n" +
                        "where exists(\n" +
                        "        select 1\n" +
                        "        from promo_codes pc\n" +
                        "        where pc.code = tt.code\n" +
                        "    )\n" +
                        "   or exists(\n" +
                        "        select 1\n" +
                        "        from promo_codes_archive pca\n" +
                        "        where pca.code = tt.code\n" +
                        "    )",
                ps -> {
                    Array arr = ps.getConnection().createArrayOf(JDBCType.VARCHAR.getName(), codes);
                    ps.setArray(1, arr);
                },
                (rs, num) -> SafePromoCode.cons(rs.getString(1)));
    }

    @Override
    public PromoCodeEntity blockCode(SafePromoCode promoCode, String reason) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("code", promoCode);
        params.put("status", PromoCodeStatus.BLOCKED.value());
        params.put("reason", reason);
        params.put("now", Instant.now());

        @Language("SQL")
        String sql = "UPDATE promo_codes " +
                " SET status = :status::promo_code_status, status_reason = :reason, status_updated_at = :now, updated_at = :now" +
                " WHERE code = :code " +
                " returning * ";


        return jdbcTemplate.queryForOption(sql, (rs, num) -> parseRow(rs), params)
                .orElseThrow(() -> new EmptyResultDataAccessException("Promo code " + promoCode + " not found", 1));
    }

    public long calculateNumOccupied(String prefix, long postfixLength) {
        return DataAccessUtils.requiredSingleResult(jdbcTemplate.query(
                "select  (select count(1) from promo_codes         where code like :prefix || '%' and length(code) = " +
                        ":totalLength ) + " +
                        "(select count(1) from promo_codes_archive where code like :prefix || '%' and length(code) = " +
                        ":totalLength)",
                (rs, num) -> rs.getLong(1),
                new MapSqlParameterSource()
                        .addValue("prefix", prefix)
                        .addValue("totalLength", prefix.length() + postfixLength)
        ));
    }

    @Override
    public PromoCodeEntity decrementRemainingActivations(SafePromoCode code) {
        @Language("SQL")
        String sql = "update promo_codes " +
                " set remaining_activations = remaining_activations - 1 " +
                " where code = ? " +
                " returning * ";

        return jdbcTemplate.queryForObject(sql, (rs, i) -> parseRow(rs), code);
    }

    @Override
    protected String pkColumnName() {
        return "code";
    }

    @Override
    public String getTableName() {
        return "promo_codes";
    }
}
