package ru.yandex.chemodan.app.psbilling.core.dao.promocodes.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.UUID;

import org.intellij.lang.annotations.Language;
import org.joda.time.Instant;
import org.springframework.dao.support.DataAccessUtils;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractPKBasedDAOImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.promocodes.PromoCodeTemplateDao;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.PromoCodeTemplateEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.PromoCodeType;
import ru.yandex.misc.db.resultSet.ResultSetExtras;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class PromoCodeTemplateDaoImpl extends AbstractPKBasedDAOImpl<PromoCodeTemplateEntity, String> implements PromoCodeTemplateDao {

    public PromoCodeTemplateDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    protected String pkColumnName() {
        return "code";
    }

    @Override
    public String getTableName() {
        return "promo_code_templates";
    }

    @Override
    public PromoCodeTemplateEntity parseRow(ResultSet rs) throws SQLException {
        ResultSetExtras extras = ResultSetExtras.cons(rs);
        return new PromoCodeTemplateEntity(
                rs.getString("code"),
                new Instant(rs.getTimestamp("created_at")),
                new Instant(rs.getTimestamp("updated_at")),
                PromoCodeType.R.fromValue(rs.getString("type")),
                Option.ofNullable(rs.getObject("user_product_price_id", UUID.class)),
                Option.ofNullable(rs.getObject("promo_template_id", UUID.class)),
                extras.getIntO("num_activations"),
                rs.getString("description"),
                rs.getString("rule_checker_bean_el"),
                new Instant(rs.getTimestamp("from_date")),
                Option.ofNullable(rs.getTimestamp("to_date")).map(Instant::new)
        );
    }

    @Override
    public PromoCodeTemplateEntity create(InsertData dataToInsert) {
        MapSqlParameterSource params = new MapSqlParameterSource()
                .addValue("code", dataToInsert.getCode())
                .addValue("now", Instant.now())
                .addValue("type", dataToInsert.getType().value())
                .addValue("user_product_price_id", dataToInsert.getUserProductPriceId().orElse((UUID) null))
                .addValue("promo_template_id", dataToInsert.getPromoTemplateId().orElse((UUID) null))
                .addValue("num_activations", dataToInsert.getNumActivations())
                .addValue("description", dataToInsert.getDescription())
                .addValue("rule_checker_bean_el", dataToInsert.getRuleCheckerBeanEl())
                .addValue("from_date", new Timestamp(dataToInsert.getFromDate().getMillis()))
                .addValue("to_date", dataToInsert.getToDate()
                        .map(Instant::getMillis)
                        .map(Timestamp::new)
                        .orElse((Timestamp) null)
                );

        @Language("SQL")
        String sql = "insert into " + getTableName() +
                " (code, created_at, updated_at, type, user_product_price_id, promo_template_id, rule_checker_bean_el," +
                " description, from_date, to_date, num_activations) " +
                " values (:code, :now, :now, :type::promo_codes_type, :user_product_price_id, :promo_template_id, " +
                " :rule_checker_bean_el, :description, :from_date, :to_date, :num_activations) " +
                " returning *";

        ListF<PromoCodeTemplateEntity> result = jdbcTemplate.query(sql, (rs, num) -> parseRow(rs), params);

        return DataAccessUtils.requiredSingleResult(result);
    }
}
