package ru.yandex.chemodan.app.psbilling.core.dao.promocodes.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Instant;
import org.springframework.dao.support.DataAccessUtils;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractPKBasedDAOImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.promocodes.UserPromoCodeDao;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.UserPromoCodeEntity;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.SafePromoCode;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class UserPromoCodeDaoImpl extends AbstractPKBasedDAOImpl<UserPromoCodeEntity, UUID> implements UserPromoCodeDao {
    public UserPromoCodeDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "user_promo_codes";
    }

    public UserPromoCodeEntity create(InsertData dataToInsert) {
        return DataAccessUtils.requiredSingleResult(jdbcTemplate.query(
                "insert into user_promo_codes " +
                        "(code, uid, created_at, order_id, user_promo_id) values " +
                        "(:code, :uid, :now, :order_id, :user_promo_id) returning *",
                (rs, num) -> parseRow(rs),
                new MapSqlParameterSource()
                        .addValue("code", dataToInsert.getCode())
                        .addValue("uid", dataToInsert.getUid())
                        .addValue("now", Instant.now())
                        .addValue("order_id", dataToInsert.getOrderId().getOrNull())
                        .addValue("user_promo_id", dataToInsert.getUserPromoId().getOrNull())
        ));
    }

    public Option<UserPromoCodeEntity> findByCodeAndUid(SafePromoCode code, PassportUid uid) {
        return jdbcTemplate.queryForOption(
                "select * from user_promo_codes where code = :code and uid = :uid",
                (rs, num) -> this.parseRow(rs),
                new MapSqlParameterSource()
                        .addValue("code", code)
                        .addValue("uid", uid.toString())
        );
    }

    @Override
    public UserPromoCodeEntity parseRow(ResultSet rs) throws SQLException {
        return new UserPromoCodeEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                SafePromoCode.cons(rs.getString("code")),
                PassportUid.cons(Long.parseLong(rs.getString("uid"))),
                Option.ofNullable(rs.getString("order_id")).map(UUID::fromString),
                Option.ofNullable(rs.getString("user_promo_id")).map(UUID::fromString)
        );
    }

    @Override
    protected String pkColumnName() {
        return "id";
    }
}
