package ru.yandex.chemodan.app.psbilling.core.dao.promos;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoPayloadEntity;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class PromoPayloadDaoImpl extends AbstractDaoImpl<PromoPayloadEntity> implements PromoPayloadDao {
    public PromoPayloadDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    private static final String PROMO_PAYLOAD_TABLE = "promo_payload";

    public String getTableName() {
        return PROMO_PAYLOAD_TABLE;
    }

    public PromoPayloadEntity parseRow(ResultSet rs) throws SQLException {
        return new PromoPayloadEntity(
                UUID.fromString(rs.getString("id")),
                UUID.fromString(rs.getString("promo_id")),
                rs.getString("payload_type"),
                rs.getString("content"),
                rs.getInt("version"),
                new Instant(rs.getTimestamp("created_at"))
        );
    }

    public MapF<UUID, PromoPayloadEntity> get(ListF<UUID> promoIds, String payloadType, Option<Integer> version) {
        if (promoIds.isEmpty()) {
            return Cf.hashMap();
        }
        MapF<UUID, PromoPayloadEntity> result = Cf.hashMap();

        jdbcTemplate.query(
            "select * from " + getTableName() +
                    " where promo_id IN (:promoIds) AND payload_type::text = :payloadType AND version = " +
                    ":version",
            (rs, num) -> parseRow(rs),
            Cf.map(
                    "promoIds", promoIds,
                    "payloadType", payloadType,
                    "version", version.orElse(PromoPayloadEntity.DEFAULT_VERSION)
            )
        ).forEach(e -> result.put(e.getPromoId(), e));

        return result;
    }

    public PromoPayloadEntity create(PromoPayloadDao.InsertData dataToInsert) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("promo_id", dataToInsert.getPromoId());
        params.put("payload_type", dataToInsert.getPayloadType());
        params.put("content", dataToInsert.getContent());
        params.put("version", dataToInsert.getVersion());
        params.put("now", now);

        return jdbcTemplate.queryForOption(
                "insert into " + getTableName() + " (" +
                        "promo_id," +
                        "payload_type," +
                        "content, " +
                        "version, " +
                        "created_at)" +
                        "values(" +
                        ":promo_id," +
                        ":payload_type::payload_type," +
                        ":content," +
                        ":version," +
                        ":now)" +
                        "RETURNING *",
                (rs, num) -> parseRow(rs), params).get();
    }

    public PromoPayloadEntity createOrUpdate(PromoPayloadDao.InsertData dataToInsert) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("promo_id", dataToInsert.getPromoId());
        params.put("payload_type", dataToInsert.getPayloadType());
        params.put("content", dataToInsert.getContent());
        params.put("version", dataToInsert.getVersion());
        params.put("now", now);

        return jdbcTemplate.queryForOption(
                "insert into " + getTableName() + " ("+
                        "promo_id, payload_type, content, " +
                        "version, created_at) " +
                   "values (" +
                        ":promo_id, :payload_type::payload_type, " +
                        ":content, :version, :now) " +
                   "on conflict (promo_id, payload_type, version) " +
                   "do update set content = :content " +
                   "returning *",
                (rs, num) -> parseRow(rs), params).get();
    }
}
