package ru.yandex.chemodan.app.psbilling.core.dao.promos;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriod;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoApplicationArea;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoApplicationType;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoTemplateEntity;
import ru.yandex.chemodan.util.exception.BadRequestException;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class PromoTemplateDaoImpl extends AbstractDaoImpl<PromoTemplateEntity> implements PromoTemplateDao {

    public PromoTemplateDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    private static final String PROMO_TEMPLATES_TABLE = "promo_templates";
    private static final String PROMO_PRODUCT_LINES_TABLE = "promo_product_lines";

    @Override
    public String getTableName() {
        return PROMO_TEMPLATES_TABLE;
    }

    @Override
    public PromoTemplateEntity parseRow(ResultSet rs) throws SQLException {
        return new PromoTemplateEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                rs.getString("code"),
                rs.getString("description"),
                new Instant(rs.getTimestamp("from_date")),
                Option.ofNullable(rs.getTimestamp("to_date")).map(Instant::new),
                PromoApplicationArea.R.fromValue(rs.getString("application_area")),
                PromoApplicationType.R.fromValue(rs.getString("application_type")),
                CustomPeriod.fromResultSet(rs, "duration_measurement", "duration"),
                Option.ofNullable(rs.getString("activation_email_template_key")),
                Option.ofNullable(rs.getString("promo_name_tanker_key_id")).map(UUID::fromString));
    }

    @Override
    public PromoTemplateEntity create(InsertData dataToInsert) {
        Instant toDate = dataToInsert.getToDate().orElse(dataToInsert.getFromDate().plus(1));
        if (dataToInsert.getFromDate().isAfter(toDate)) {
            throw new BadRequestException("to_date can't be before from_date");
        }

        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("description", dataToInsert.getDescription());
        params.put("from_date", dataToInsert.getFromDate());
        params.put("to_date", dataToInsert.getToDate().getOrNull());
        params.put("application_area", dataToInsert.getApplicationArea().value());
        params.put("application_type", dataToInsert.getApplicationType().value());
        CustomPeriod duration = dataToInsert.getDuration().getOrNull();
        params.put("duration", duration != null ? duration.getValue() : null);
        params.put("duration_measurement", duration != null ? duration.getUnit().value() : null);
        params.put("activation_email_template_key", dataToInsert.getActivationEmailTemplate().getOrNull());
        params.put("promo_name_tanker_key_id", dataToInsert.getPromoNameTankerKey().getOrNull());
        params.put("code", dataToInsert.getCode());
        params.put("now", now);

        return jdbcTemplate.queryForOption(
                        "insert into " + PROMO_TEMPLATES_TABLE + " (" +
                                "created_at," +
                                "description," +
                                "code," +
                                "from_date, " +
                                "to_date, " +
                                "application_area, " +
                                "application_type, " +
                                "duration, " +
                                "duration_measurement, " +
                                "activation_email_template_key, " +
                                "promo_name_tanker_key_id) " +
                                "values(" +
                                ":now, " +
                                ":description, " +
                                ":code, " +
                                ":from_date, " +
                                ":to_date, " +
                                ":application_area::promo_application_area, " +
                                ":application_type::promo_application_type, " +
                                ":duration, " +
                                ":duration_measurement, " +
                                ":activation_email_template_key, " +
                                ":promo_name_tanker_key_id::uuid) " +
                                "RETURNING *",
                        (rs, num) -> parseRow(rs),
                        params
                )
                .get();
    }

    @Override
    public void bindProductLines(UUID promoId, Collection<UUID> productLines) {
        jdbcTemplate.batchUpdate(
                "insert into " + PROMO_PRODUCT_LINES_TABLE + " (promo_template_id, product_line_id)" +
                        " values (?, ?)" +
                        " on conflict (promo_template_id, product_line_id) DO NOTHING",
                Cf.toList(productLines).map(t -> new Object[]{promoId, t}));
    }

    @Override
    public SetF<UUID> getProductLines(UUID promoId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("promoId", promoId);

        return jdbcTemplate.query(
                        "select product_line_id " +
                                "from " + PROMO_PRODUCT_LINES_TABLE + " " +
                                "where promo_template_id = :promoId",
                        (rs, num) -> UUID.fromString(rs.getString("product_line_id")),
                        params
                )
                .unique();
    }

    @Override
    public ListF<PromoTemplateEntity> findByIds(ListF<UUID> promoTemplateIds) {
        if (promoTemplateIds.isEmpty()) {
            return Cf.list();
        }

        return jdbcTemplate.query(
                "select * from " + PROMO_TEMPLATES_TABLE +
                        " where id in (:promoTemplateIds)",
                (rs, num) -> parseRow(rs), Cf.map("promoTemplateIds", promoTemplateIds));
    }

    @Override
    public Option<PromoTemplateEntity> findByCode(String code) {
        return jdbcTemplate.queryForOption(
                "select * " +
                        "from " + PROMO_TEMPLATES_TABLE + " " +
                        "where code = :code",
                (rs, num) -> parseRow(rs), Cf.map("code", code));
    }

    @Override
    public MapF<PromoTemplateEntity, SetF<UUID>> findOnlyB2cByProductLines(ListF<UUID> productLines) {
        if (productLines.isEmpty()) {
            return Cf.map();
        }
        MapF<String, Object> params = Cf.hashMap();
        params.put("productLines", productLines);
        params.put("area", PromoApplicationArea.b2cList.map(PromoApplicationArea::value));

        return jdbcTemplate.query(
                        "select distinct p.*, pl.product_line_id " +
                                "from " + PROMO_TEMPLATES_TABLE + " as p " +
                                "join " + PROMO_PRODUCT_LINES_TABLE + " as pl on p.id = pl.promo_template_id " +
                                "where p.id in (select distinct promo_template_id from " + PROMO_PRODUCT_LINES_TABLE +
                                "     where product_line_id in (:productLines))" +
                                "   and p.application_area::text in (:area)",
                        (rs, num) -> Tuple2.tuple(parseRow(rs), rs.getString("product_line_id")),
                        params
                )
                .groupBy(tuple -> tuple._1)
                .mapValues(tuplesList -> tuplesList.map(Tuple2::get2).filterNotNull().map(UUID::fromString).unique());
    }


    @Override
    public MapF<PromoTemplateEntity, SetF<UUID>> findOnlyB2cWithProductLines() {
        return jdbcTemplate.query(
                        "select distinct p.*, pl.product_line_id " +
                                "from " + PROMO_TEMPLATES_TABLE + " as p " +
                                "left join " + PROMO_PRODUCT_LINES_TABLE + " as pl on p.id = pl.promo_template_id " +
                                "where p.application_area::text in (:area)",
                        (rs, num) -> Tuple2.tuple(parseRow(rs), rs.getString("product_line_id")),
                        Cf.map("area", PromoApplicationArea.b2cList.map(PromoApplicationArea::value))
                )
                .groupBy(tuple -> tuple._1)
                .mapValues(tuplesList -> tuplesList.map(Tuple2::get2).filterNotNull().map(UUID::fromString).unique());
    }
}
